/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/

#include "MAX4822.h"

//*********************************************************************
MAX4822::MAX4822(SPI & spi_bus, PinName cs, uint8_t num_devices): 
m_spi(spi_bus), m_cs(cs, 1), m_num_devices(num_devices)
{
    if(m_num_devices)
    {
        for(uint8_t idx = 0; idx < m_num_devices; idx++)
        {
            m_relay_data[idx] = 0;
            m_pwr_save_data[idx] = 0;
        }
    }
    else
    {
        m_relay_data[0] = 0;
        m_pwr_save_data[0] = 0;
    }
}

//*********************************************************************
MAX4822::~MAX4822()
{
}

//*********************************************************************
MAX4822::CmdResult MAX4822::set_all_relays(DigitalOut & set, uint8_t device)
{
    MAX4822::CmdResult result = OpFailure;
    
    if(device <= m_num_devices)
    {
        set = 0;
        wait_us(1);
        set = 1;
        
        m_relay_data[device] = 0xFF;
        result = MAX4822::Success;
    }
    
    return result;
}

//*********************************************************************
MAX4822::CmdResult MAX4822::reset_all_relays(DigitalOut & reset, uint8_t device)
{
    MAX4822::CmdResult result = OpFailure;
    
    if(device <= m_num_devices)
    {
        reset = 0;
        wait_us(1);
        reset = 1;
        
        m_relay_data[device] = 0;
        result = MAX4822::Success;
    }
    
    return result;
    
}

//*********************************************************************
MAX4822::CmdResult MAX4822::set_relay(RelayChannel r, bool send_data, uint8_t n)
{
    MAX4822::CmdResult result = OpFailure;
    
    if(n <= m_num_devices)
    {
        m_relay_data[n] |= (1 << (r - 1));
        
        if(send_data)
        {
            uint16_t num_writes = n + 1;
            
            m_cs = 0;
            while(num_writes--)
            {
                m_spi.write(MAX4822::OUTPUT_CNTL_REG);
                m_spi.write(m_relay_data[num_writes]);
            }
            m_cs = 1;
        }
        
        result = MAX4822::Success;
    }
    
    return result;
}

//*********************************************************************
MAX4822::CmdResult MAX4822::reset_relay(RelayChannel r, bool send_data, uint8_t n)
{
    MAX4822::CmdResult result = OpFailure;
    
    if(n <= m_num_devices)
    {
        m_relay_data[n] &= ~(1 << (r - 1));
        
        if(send_data)
        {
            uint16_t num_writes = n + 1;
            
            m_cs = 0;
            while(num_writes--)
            {
                m_spi.write(MAX4822::OUTPUT_CNTL_REG);
                m_spi.write(m_relay_data[num_writes]);
            }
            m_cs = 1;
        }
        
        result = MAX4822::Success;
    }
    
    return result;
}

//*********************************************************************
MAX4822::CmdResult MAX4822::set_pwr_save(PowerSave pwr_save, bool send_data, uint8_t n)
{
    MAX4822::CmdResult result = OpFailure;
    
    if(n <= m_num_devices)
    {
        m_pwr_save_data[n] = pwr_save;
        
        if(send_data)
        {
            uint16_t num_writes = n + 1;
            
            m_cs = 0;
            while(num_writes--)
            {
                m_spi.write(MAX4822::POWER_SAVE_REG);
                m_spi.write(m_pwr_save_data[num_writes]);
            }
            m_cs = 1;
        }
        
        result = MAX4822::Success;
    }
    
    return result;
}
