/******************************************************************//**
* @file max14871_shield.h
* Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#ifndef MAX14871_SHIELD_H
#define MAX14871_SHIELD_H

#include "mbed.h"
#include "max7300.h"
#include "max5387.h"


/**
* @brief MAX14871 Full-Bridge DC Motor Driver MBED Shield Library
*
* @details The MAXREFDES89# provides an ideal interface for anyone
* developing with brushed DC motors. The design is an 
* mbed®-compatible, Arduino® form factor shield for the rapid 
* development of brushed DC motor applications. The shield contains 
* four MAX14871 full-bridge DC motor drivers for driving up to 4 
* motors. A MAX17501 DC-DC converter allows the system to operate 
* from a single 7 to 36VDC power supply. Four MAX4704 4:1 
* multiplexers permit setting the current regulation mode of the 
* MAX14871, while two MAX5387 digital potentiometers provide the 
* ability to set the motor current limit. A MAX7300 GPIO expander 
* supports interfacing each motor driver circuit to the 
* mbed-enabled microcontroller platform.
*
* @code
* @endcode
*/
class Max14871_Shield
{
    public:
    
    /**
    * @brief Motor driver channels
    * @details Simple enumeration of channels available on the shield
    */
    typedef enum
    {
        MD1 = 1,
        MD2,
        MD3,
        MD4
    }max14871_motor_driver_t;
    
    
    /**
    * @brief Operating modes
    * @details Motor operating modes offered on each channel
    */
    typedef enum
    {
        COAST = 1,
        BRAKE,
        REVERSE,
        FORWARD
    }max14871_operating_mode_t;
    
    
    /**
    * @brief Current regulation modes
    * @details Three different current regulation modes offered with
    * either an internal or external reference for setting the regulation limit.
    * This is not the fault current which is fixed, see datasheet.
    */
    typedef enum
    {
        RIPPLE_25_INTERNAL_REF = 1,
        RIPPLE_25_EXTERNAL_REF,
        TCOFF_FAST_INTERNAL_REF,
        TCOFF_SLOW_INTERNAL_REF,
        TCOFF_FAST_EXTERNAL_REF,
        TCOFF_SLOW_EXTERNAL_REF
    }max14871_current_regulation_mode_t;
    
    
    /**********************************************************//**
    * @brief Constructor for Max14871_Shield Class.  
    * 
    * @details Allows user to use existing I2C object
    *
    * On Entry:
    *     @param[in] i2c_bus - pointer to existing I2C object
    *     @param[in] default_config - true if board uses default
    *                                 pwm channels and I2C addressing
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    Max14871_Shield(I2C *i2c_bus, bool default_config = true);
    
    
    /**********************************************************//**
    * @brief Constructor for Max14871_Shield Class.  
    * 
    * @details Allows user to create a new I2C object if not 
    *          already using one
    *
    * On Entry:
    *     @param[in] sda - sda pin of I2C bus
    *     @param[in] scl - scl pin of I2C bus
    *     @param[in] default_config - true if board uses default
    *                                 pwm channels and I2C addressing
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    Max14871_Shield(PinName sda, PinName scl, bool default_config = true);
    
    
    /**********************************************************//**
    * @brief Default destructor for Max14871_Shield Class.  
    *
    * @details Destroys I2C object if owner 
    *
    * On Entry:
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    ~Max14871_Shield();
    
    
    /**********************************************************//**
    * @brief Set the operating mode of the motor driver
    *
    * @details Configures the /EN and DIR pins of the motor driver
    *  via the MAX7300 GPIO Expander   
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] mode - 1 of 4 operating modes of the motor driver 
    *
    * On Exit:
    *
    * @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_operating_mode(max14871_motor_driver_t md, 
                               max14871_operating_mode_t mode);
    
    
    /**********************************************************//**
    * @brief 
    *
    * @details    
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] mode - 1 of 6 current regulation modes of the 
    *                      motor driver
    *    @param[in] vref - sets maximum motor current, Max of 2.0V
    *
    * On Exit:
    *
    * @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_current_regulation_mode(max14871_motor_driver_t md, 
                                        max14871_current_regulation_mode_t mode,
                                        float vref = 1.0);
    
    
    /**********************************************************//**
    * @brief Sets pwm channel for given motor driver
    *
    * @details Must use default, or alternate channel for specific 
    *          motor driver.  Function allows for mix of default and 
    *          alternates for each motor driver vs all default or 
    *          all alternate.
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] ch - PWM channel using Arduino naming convention
    *
    * On Exit:
    *
    * @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_pwm_channel(max14871_motor_driver_t md, PinName ch);
    
    
    /**********************************************************//**
    * @brief Sets period of pwm signal for selected motor driver
    *
    * @details period must be in micro-seconds
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] period - PWM period specified in seconds
    *
    * On Exit:
    *
    * @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_pwm_period(max14871_motor_driver_t md, float period);
    
    
    /**********************************************************//**
    * @brief Sets duty cycle of pwm signal for selected motor driver
    *
    * @details duty cycle must be in micro-seconds
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] duty_cycle - duty cycle of the pwm signal specified
    *                            as a percentage
    *
    * On Exit:
    *
    * @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_pwm_duty_cycle(max14871_motor_driver_t md, float duty_cycle);
    
    
    /**********************************************************//**
    * @brief Get operating mode of selected motor driver
    *
    * @details 
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    
    * On Exit:
    *
    * @return Operating mode of selected motor driver
    **************************************************************/
    max14871_operating_mode_t get_operating_mode(max14871_motor_driver_t md);
    
    
    /**********************************************************//**
    * @brief Get current regulation mode of selected motor driver
    *
    * @details 
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    
    * On Exit:
    *
    * @return Current regulation mode of selected motor driver
    **************************************************************/
    max14871_current_regulation_mode_t get_current_regulation_mode(max14871_motor_driver_t md);
    
    
    /**********************************************************//**
    * @brief Get duty cycle of selected motor driver
    *
    * @details 
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    
    * On Exit:
    *
    * @return Duty cycle of selected motor driver
    **************************************************************/
    float get_pwm_duty_cycle(max14871_motor_driver_t md);
    
    
    /**********************************************************//**
    * @brief Get pwm period of selected motor driver
    *
    * @details 
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    
    * On Exit:
    *
    * @return pwm period of selected motor driver
    **************************************************************/
    float get_pwm_period(max14871_motor_driver_t md);
    
    
    /**********************************************************//**
    * @brief Get external voltage reference of selected motor driver
    *
    * @details 
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    
    * On Exit:
    *
    * @return External voltage reference of selected motor driver
    **************************************************************/
    float get_external_voltage_ref(max14871_motor_driver_t md);
                                        
    
    private:
    
    struct motor_data_s
    {
        max14871_operating_mode_t op_mode;
        max14871_current_regulation_mode_t i_reg_mode;
        float duty_cycle;
        float period;
        float v_ref;
    };
    
    struct motor_data_s _motor_data_array[4];
    
    I2C *_p_i2c;
    Max7300 *_p_io_expander;
    Max5387 *_p_digi_pot1;
    Max5387 *_p_digi_pot2;
    PwmOut *_p_pwm1;
    PwmOut *_p_pwm2;
    PwmOut *_p_pwm3;
    PwmOut *_p_pwm4;
    
    bool _i2c_owner;
    
    void init_board(void);
};

#endif/* MAX14871_SHIELD_H */