/******************************************************************//**
* @file max14871_shield.h
*
* @author Justin Jordan
*
* @version 0.0
*
* Started: 18JUL15
*
* Updated: 
*
* @brief Header file for Max14871_Shield class
***********************************************************************
* Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#ifndef MAX14871_SHIELD_H
#define MAX14871_SHIELD_H

#include "mbed.h"
#include "max7300.h"
#include "max5387.h"


class Max14871_Shield
{
    public:
    
    typedef enum
    {
        MD1 = 1,
        MD2,
        MD3,
        MD4
    }max14871_motor_driver_t;
    
    
    typedef enum
    {
        COAST = 1,
        BRAKE,
        REVERSE,
        FORWARD
    }max14871_operating_mode_t;
    
    
    typedef enum
    {
        RIPPLE_25_INTERNAL_REF = 1,
        RIPPLE_25_EXTERNAL_REF,
        TCOFF_FAST_INTERNAL_REF,
        TCOFF_SLOW_INTERNAL_REF,
        TCOFF_FAST_EXTERNAL_REF,
        TCOFF_SLOW_EXTERNAL_REF
    }max14871_current_regulation_mode_t;
    
    
    /**********************************************************//**
    * @brief Constructor for Max14871_Shield Class.  
    * 
    * @details Allows user to use existing I2C object
    *
    * On Entry:
    *     @param[in] i2c_bus - pointer to existing I2C object
    *     @param[in] default_config - true if board uses default
    *                                 pwm channels and I2C addressing
    *
    * On Exit:
    *    @return none
    **************************************************************/
    Max14871_Shield(I2C *i2c_bus, bool default_config);
    
    
    /**********************************************************//**
    * @brief Constructor for Max14871_Shield Class.  
    * 
    * @details Allows user to create a new I2C object if not 
    *          already using one
    *
    * On Entry:
    *     @param[in] sda - sda pin of I2C bus
    *     @param[in] scl - scl pin of I2C bus
    *     @param[in] default_config - true if board uses default
    *                                 pwm channels and I2C addressing
    *
    * On Exit:
    *    @return none
    **************************************************************/
    Max14871_Shield(PinName sda, PinName scl, bool default_config);
    
    
    /**********************************************************//**
    * @brief Default destructor for Max14871_Shield Class.  
    *
    * @details Destroys I2C object if owner 
    *
    * On Entry:
    *
    * On Exit:
    *    @return none
    **************************************************************/
    ~Max14871_Shield();
    
    
    /**********************************************************//**
    * @brief Set the operating mode of the motor driver
    *
    * @details Configures the /EN and DIR pins of the motor driver
    *  via the MAX7300 GPIO Expander   
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] mode - 1 of 4 operating modes of the motor driver 
    *
    * On Exit:
    *    @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_operating_mode(max14871_motor_driver_t md, 
                               max14871_operating_mode_t mode);
    
    
    /**********************************************************//**
    * @brief 
    *
    * @details    
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] mode - 1 of 6 current regulation modes of the 
    *                      motor driver
    *    @param[in] vref - sets maximum motor current, Max of 2.0V
    *
    * On Exit:
    *    @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_current_regulation_mode(max14871_motor_driver_t md, 
                                        max14871_current_regulation_mode_t mode,
                                        float vref);
    
    
    /**********************************************************//**
    * @brief Sets period of pwm signal for selected motor driver
    *
    * @details period must be in micro-seconds
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] period - PWM period specified in seconds
    *
    * On Exit:
    *    @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_pwm_period(max14871_motor_driver_t md, float period);
    
    
    /**********************************************************//**
    * @brief Sets duty cycle of pwm signal for selected motor driver
    *
    * @details duty cycle must be in micro-seconds
    *
    * On Entry:
    *    @param[in] md - 1 of 4 motor drivers on the shield
    *    @param[in] duty_cycle - duty cycle of the pwm signal specified
    *                            as a percentage
    *
    * On Exit:
    *    @return 0 on success, non-0 on failure
    **************************************************************/
    int16_t set_pwm_duty_cycle(max14871_motor_driver_t md, float duty_cycle);
                                        
    
    private:
    
    I2C *_p_i2c;
    Max7300 *_p_io_expander;
    Max5387 *_p_digi_pot1;
    Max5387 *_p_digi_pot2;
    PwmOut *_p_pwm1;
    PwmOut *_p_pwm2;
    PwmOut *_p_pwm3;
    PwmOut *_p_pwm4;
    
    bool _i2c_owner;
    
    void init_board(void);
};

#endif/* MAX14871_SHIELD_H */