/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef SENSORNODE_HPP
#define SENSORNODE_HPP

#include <stdint.h>
#include <MaximInterface/Devices/DS28E15_22_25.hpp>
#include <MaximInterface/Utilities/RomId.hpp>
#include <MaximInterface/Utilities/ManId.hpp>
#include "DS7505.hpp"
#include "MAX44009.hpp"

class SensorData;
namespace MaximInterface { class DS2465; }
namespace mbed { class I2C; }

/// Interface to the Authenticated Sensor Node peripheral board.
class SensorNode {
public:
  enum State {
    UnableToCommunicate, ///< I2C or 1-Wire communication failure.
    NotProvisioned,      ///< DS28E15 has not been provisioned.
    NotAuthentic,        ///< DS28E15 is not authentic.
    Authentic            ///< DS218E15 is authentic.
  };

  /// @param i2c Configured I2C communication interface.
  /// @param ds7505_i2c_addr I2C bus address of the DS7505 in mbed format.
  /// @param max44009_i2c_addr I2C bus address of the MAX44009 in mbed format.
  /// @param ds2465 Interface to DS2465 on the Controller.
  SensorNode(mbed::I2C & i2c, uint8_t ds7505_i2c_addr,
             uint8_t max44009_i2c_addr, MaximInterface::DS2465 & ds2465);

  /// Detect if an authentic Sensor Node is connected.
  /// @param userEntropy Additional entropy to supply to the RNG.
  /// @returns Authentic on success.
  State detect(unsigned int userEntropy);

  /// Read the current temperature and filter life measurements with authentication.
  /// @param userEntropy Additional entropy to supply to the RNG.
  /// @param sensorData Measurements output location.
  /// @returns Authentic on success.
  State authenticatedReadSensorData(unsigned int userEntropy,
                                    SensorData & sensorData);

  /// Reads the current temperature and filter life measurements.
  /// @param sensorData Measurements output location.
  /// @returns True on success or false if unable to communicate with a sensor.
  bool readSensorData(SensorData & sensorData);

  /// Get the ROM ID for this sensor node.
  const MaximInterface::RomId romId() const { return romId_; }

  /// Get the initial lux measurement for this sensor node.
  double initialLux() const { return initialLux_; }

  // Grant access to hardware interfaces for provisioning.
  friend bool provisionCoprocessor(MaximInterface::DS2465 & ds2465);
  friend bool provisionSensorNode(SensorNode & sensorNode, bool validSecret);

private:
  /// Authenticated data stored in DS28E15.
  struct AuthData {
    static const uint8_t initialFilterLife = 100;
    MaximInterface::DS28E15::Segment segment;
    int pageNum, segmentNum;

    uint8_t & filterLife() { return segment[0]; }
    const uint8_t & filterLife() const { return segment[0]; }

    AuthData() : pageNum(0), segmentNum(0) { reset(); }

    void reset() {
      segment.fill(0);
      filterLife() = initialFilterLife;
    }
  };

  /// Padding value used in creation of the Slave Secret.
  static const uint8_t defaultPaddingByte = 0x00;

  static const MaximInterface::ManId manId;

  /// Indicates hardware RNG is initialized.
  static bool rngInitialized;
  // Initialize the hardware RNG.
  static void initializeRng();

  /// Initial lux measurement taken on initialization.
  /// Assumed to be the maximum intensity that will be observed.
  double initialLux_;

  /// Authenticated data stored on the DS28E15.
  AuthData authData;

  // Hardware interfaces
  MaximInterface::DS2465 & ds2465; ///< Interface to DS2465 on Controller.
  MaximInterface::DS28E15 ds28e15; ///< DS28E15 for authentication.
  MaximInterface::RomId romId_;
  DS7505 ds7505;     ///< DS7505 temperature sensor.
  MAX44009 max44009; ///< MAX44009 optical light sensor.

  /// Initialize sensors for measurement.
  /// @returns True on success.
  bool initializeSensors();

  /// Select the Slave Secret for this Sensor Node in the Controller.
  /// @returns True on success.
  bool setSecret();

  /// Check if the Sensor Board is provisioned.
  /// @param provisioned True if the sensor board is provisioned.
  /// @returns True if provisioning check was successful.
  bool checkProvisioned(bool & provisioned);

  /// Check if the Sensor Board is authentic.
  /// @param userEntropy Additional entropy to supply to the RNG.
  /// @returns True if the Sensor Board passed the authentication check.
  bool checkAuthentic(unsigned int userEntropy);

  /// Checks if the authenticated data stored in the DS28E15 needs to be updated.
  /// Updates the authenticated data if necessary.
  /// @param sensorData Current sensor data to check.
  /// @returns True on success.
  bool checkAndWriteAuthData(SensorData & sensorData);
};

#endif