/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef WEBSERVERINTERFACE_HPP
#define WEBSERVERINTERFACE_HPP

#include <string>
#include <MaximInterface/Utilities/RomId.hpp>

/// The message type described by the POST.
enum PostEvent {
  SensorDataEvent,   ///< Adding sensor data to the log.
  InvalidSensorEvent ///< Reporting an invalid sensor node.
};

namespace mbed { class Serial; }
namespace MaximInterface { class DS2465; }
class ESP8266;
struct SensorData;

/// Network interface to the web server supporting authenticated posting of event
/// through an HTTP challenge-respones scheme with SHA-256 data signing.
class WebServerInterface {
public:
  /// @param esp8266 Interface to ESP8266 for Wi-Fi access.
  /// @param pc Optional serial interface for received web traffic.
  WebServerInterface(ESP8266 & esp8266) : esp8266(esp8266) {}

  /// Initialize network interface and connect to access point.
  /// @returns True on success.
  bool initialize();

  /// Send an authenticated event message to the web server.
  /// @param macCoProc
  /// Coprocessor such as the DS2465 used to calculate the authentication MAC.
  /// @param event Event message type.
  /// @postData Message body as determined by the event message type.
  /// @setSecret True if the Transport Secret needs to be selected in the coprocessor.
  /// @returns True on success.
  bool authPostHttpEvent(MaximInterface::DS2465 & macCoproc, PostEvent event,
                         const std::string & postData, bool setSecret);

  /// Format sensor data as text suitable for use in a POST body.
  /// @param sensorData Sensor data to format.
  /// @returns Data formatted for web server.
  static std::string formatSensorDataPostBody(const SensorData & sensorData);

  /// @{
  /// Session ID used by the web server to distinguish between multiple Controllers.
  const MaximInterface::RomId & sessionId() const { return m_sessionId; }
  const std::string & sessionIdString() const { return m_sessionIdString; }
  void setSessionId(const MaximInterface::RomId & sessionId);
  /// @}

private:
  /// @{
  /// Configuration strings.
  static const char wifiSsid[];
  static const char wifiPassword[];
  static const char serverAddress[];
  static const unsigned int serverPort;
  static const char serverPostPath[];
  static const char serverChallengePath[];
  /// @}

  MaximInterface::RomId m_sessionId;
  std::string m_sessionIdString;

  ESP8266 & esp8266;
};

#endif