/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef DS7505_HPP
#define DS7505_HPP

#include <stdint.h>

namespace mbed { class I2C; }

/// Interface to the DS7505 temperature sensor.
class DS7505 {
public:
  enum Result { Success, Hardware_Failure, Out_of_Range };

  /// @param I2C_interface A configured I2C interface to use for communication.
  /// @param I2C_address Device bus address in mbed format.
  DS7505(mbed::I2C & I2C_interface, uint8_t I2C_address);

  /// Set the resolution for temperature conversions.
  /// @param resolution Number of decimal bit from 1 to 4.
  Result set_resolution(uint8_t resolution);

  /// Reads the current temperature with an exponent of 10^-2.
  /// @note Compatible with Bluetooth characteristic:
  /// org.bluetooth.characteristic.temperature.
  Result read_current_temp(int16_t & temperature) const;

  /// Reads the current temperature as a floating point value.
  Result read_current_temp(double & temperature) const;

  /// Reads the current temperature as an integer value.
  Result read_current_temp(int8_t & temperature) const;

private:
  /// Bit resolution of temperature conversions.
  enum Config_Resolution {
    Config_9b_Res = 0x00,
    Config_10b_Res = 0x20,
    Config_11b_Res = 0x40,
    Config_12b_Res = 0x60
  };

  /// DS7505 Register addresses.
  enum Register {
    Temperature_Reg = 0x00,
    Configuration_Reg = 0x01,
    Thyst_Reg = 0x02,
    Tos_Reg = 0x03
  };

  /// Represents a DS7505 configuration.
  struct Config {
    Config_Resolution resolution;
    bool enable_shutdown_mode;

    Config(Config_Resolution resolution, bool enable_shutdown_mode)
        : resolution(resolution), enable_shutdown_mode(enable_shutdown_mode) {}
  };

  /// @note Mark as mutable to allow manipulation by read_temp_sensor().
  mutable Config m_current_config;

  mbed::I2C & m_I2C_interface;
  uint8_t m_I2C_address;

  /// Returns the maximum time needed in milliseconds for a sample at the
  /// specified resolution.
  static uint8_t get_measure_delay_ms(Config_Resolution resolution);

  /// Reads the current temperature via I2C. Assumes that the I2C register
  /// pointer is already set to the temperature register.
  /// @param sensor_data
  /// Output for raw data from DS7505 with upper and lower bytes combined.
  /// @returns True on success.
  bool read_temp_sensor_data(uint16_t & sensor_data) const;

  /// Reads the current temperature with support for shutdown mode.
  /// @param sensor_data Output for raw data from DS7505 with upper and lower
  /// bytes combined.
  /// @returns Success or Hardware_Failure.
  Result read_temp_sensor(uint16_t & sensor_data) const;

  /// Sets the I2C register pointer for the next operation.
  /// @param pointer_reg Desired register to set.
  /// @returns True on success.
  /// @note Allow marking const since not public.
  bool set_register_pointer(Register pointer_reg) const;

  /// Writes to a device register via I2C.
  /// @param write_reg Register to write to.
  /// @param write_val Value to write to the register.
  /// @returns True on success.
  /// @note Allow marking const since not public
  bool write_register(Register write_reg, uint8_t write_val) const;

  /// Writes the current configuration via I2C.
  /// @returns True on success.
  /// @note Allow marking const since not public
  bool write_current_config() const;
};

#endif
