/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <MaximInterface/Devices/DS2465.hpp>
#include <MaximInterface/Devices/DS28E15_22_25.hpp>
#include <MaximInterface/Utilities/Segment.hpp>
#include "Factory.hpp"
#include "SensorNode.hpp"

using namespace MaximInterface;

/// The valid master secret for the system.
static const Sha256::Hash masterSecret = {
    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
    0x1C, 0x1D, 0x1E, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
    0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x31, 0x32};
/// An invalid master secret for example purposes.
static const Sha256::Hash invalidMasterSecret = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

bool provisionCoprocessor(DS2465 & ds2465) {
  bool result = !ds2465.writeMasterSecret(masterSecret);
  if (result) {
    SensorNode::AuthData authData;
    DS28E15::Page pageData;
    pageData.fill(uint8_t(SensorNode::defaultPaddingByte));
    const std::pair<DS28E15::Page::iterator, DS28E15::Page::iterator>
        pageSegment =
            createSegment(pageData.begin(), pageData.end(),
                          authData.segment.size(), authData.segmentNum);
    if (pageSegment.first != pageData.end()) {
      std::copy(authData.segment.begin(), authData.segment.end(),
                pageSegment.first);
    }
    result = !ds2465.writePage(0, pageData);
  }
  return result;
}

bool provisionSensorNode(SensorNode & sensorNode, bool validSecret) {
  const int blockNum = sensorNode.authData.pageNum / 2;
  const DS28E15::BlockProtection desiredProtection(
      false, false, false, true, blockNum); // Authentication Protection only

  // Reset to starting defaults
  sensorNode.authData.reset();

  // Read current protection status
  DS28E15::BlockProtection protectionStatus;
  bool result =
      !sensorNode.ds28e15.readBlockProtection(blockNum, protectionStatus);
  // Check if invalid protections are set
  if (result)
    result = ((protectionStatus.statusByte() &
               ~(desiredProtection.statusByte())) == 0x00);
  // Load secret into scratchpad
  if (result)
    result = !sensorNode.ds28e15.writeScratchpad(
        validSecret ? masterSecret : invalidMasterSecret);
  // Load master secret from scratchpad without locking
  if (result)
    result = !sensorNode.ds28e15.loadSecret(false);

  // Setup is complete if not using a valid secret
  if (!validSecret)
    return result;

  // Create constant partial secret
  DS28E15::Scratchpad partialSecret;
  DS28E15::Page pageData;
  partialSecret.fill(uint8_t(SensorNode::defaultPaddingByte));

  // Read page data
  if (result)
    result =
        !sensorNode.ds28e15.readPage(sensorNode.authData.pageNum, pageData);
  // Load partial secret into scratchpad
  if (result)
    result = !sensorNode.ds28e15.writeScratchpad(partialSecret);
  // Compute secret
  if (result)
    result =
        !sensorNode.ds28e15.computeSecret(sensorNode.authData.pageNum, false);
  // Configure slave secret on DS2465
  if (result) {
    const Sha256::SlaveSecretData data = DS28E15::createSlaveSecretData(
        pageData, sensorNode.authData.pageNum, partialSecret,
        sensorNode.romId(), sensorNode.manId);
    result = !sensorNode.ds2465.computeSlaveSecret(data);
  }
  // Enable authentication protection if not set
  if (result && (protectionStatus != desiredProtection)) {
    const Sha256::WriteMacData data = DS28E15::createProtectionWriteMacData(
        desiredProtection, protectionStatus, sensorNode.romId(),
        sensorNode.manId);
    Sha256::Hash mac;
    result = !sensorNode.ds2465.computeWriteMac(data, mac);
    if (result)
      result =
          !sensorNode.ds28e15.writeAuthBlockProtection(desiredProtection, mac);
  }
  // Write initial filter life and set all other segments to default value
  if (result) {
    DS28E15::Segment blankSegment;
    blankSegment.fill(uint8_t(SensorNode::defaultPaddingByte));
    for (int i = 0; i < DS28E15::segmentsPerPage; i++) {
      const DS28E15::Segment newSegment = (i == sensorNode.authData.segmentNum)
                                              ? sensorNode.authData.segment
                                              : blankSegment;
      const std::pair<DS28E15::Page::iterator, DS28E15::Page::iterator>
          pageSegment = createSegment(pageData.begin(), pageData.end(),
                                      DS28E15::Segment::size(), i);
      DS28E15::Segment oldSegment;
      std::copy(pageSegment.first, pageSegment.second, oldSegment.begin());
      const Sha256::WriteMacData data = DS28E15::createSegmentWriteMacData(
          sensorNode.authData.pageNum, i, newSegment, oldSegment,
          sensorNode.romId(), sensorNode.manId);
      Sha256::Hash mac;
      result = !sensorNode.ds2465.computeWriteMac(data, mac);
      if (!result)
        break;
      result = !sensorNode.ds28e15.writeAuthSegment(sensorNode.authData.pageNum,
                                                    i, newSegment, mac);
      if (!result)
        break;
    }
  }

  // Reload secret with known page values
  // Load master secret into scratchpad
  if (result)
    result = !sensorNode.ds28e15.writeScratchpad(masterSecret);
  // Load master secret
  if (result)
    result = !sensorNode.ds28e15.loadSecret(false);
  // Read page data
  if (result)
    result =
        !sensorNode.ds28e15.readPage(sensorNode.authData.pageNum, pageData);
  // Write partial secret to scratchpad
  if (result)
    result = !sensorNode.ds28e15.writeScratchpad(partialSecret);
  // Compute secret
  if (result)
    result =
        !sensorNode.ds28e15.computeSecret(sensorNode.authData.pageNum, false);
  // Configure slave secret on DS2465
  if (result) {
    const Sha256::SlaveSecretData data = DS28E15::createSlaveSecretData(
        pageData, sensorNode.authData.pageNum, partialSecret,
        sensorNode.romId(), sensorNode.manId);
    result = !sensorNode.ds2465.computeSlaveSecret(data);
  }

  return result;
}