/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <I2C.h>
#include <MaximInterface/Devices/DS2465.hpp>
#include <MaximInterface/Links/RomCommands.hpp>
#include <MaximInterface/Platforms/mbed/Sleep.hpp>
#include "SensorNode.hpp"
#include "SensorData.hpp"

#ifdef TARGET_MAX32600
#include <max32600.h>
#include <clkman_regs.h>
#include <tpu_regs.h>
#else
#include <cstdlib>
#endif

using namespace MaximInterface;

const ManId SensorNode::manId = {0, 0};
bool SensorNode::rngInitialized = false;

void SensorNode::initializeRng() {
#ifdef TARGET_MAX32600
  // Enable crypto oscillator
  MXC_CLKMAN->clk_config |= (MXC_F_CLKMAN_CLK_CONFIG_CRYPTO_ENABLE |
                             MXC_F_CLKMAN_CLK_CONFIG_CRYPTO_RESET_N);
  // Wait for crypto oscillator stability
  while ((MXC_CLKMAN->intfl & MXC_F_CLKMAN_INTFL_CRYPTO_STABLE) !=
         MXC_F_CLKMAN_INTFL_CRYPTO_STABLE);
  // Disable crypto clock gating
  MXC_CLKMAN->clk_ctrl |= MXC_F_CLKMAN_CLK_CTRL_CRYPTO_GATE_N;
  // Set PRNG clock to crypto clock
  MXC_CLKMAN->crypt_clk_ctrl_2_prng = MXC_CLKMAN->clk_ctrl_10_prng = 1;
  // Use dynamic clock gating
  MXC_CLKMAN->clk_gate_ctrl2 |=
      (1 << MXC_F_CLKMAN_CLK_GATE_CTRL2_TPU_CLK_GATER_POS);
#endif
}

SensorNode::SensorNode(::mbed::I2C & i2c, uint8_t ds7505_i2c_addr,
                       uint8_t max44009_i2c_addr, DS2465 & ds2465)
    : initialLux_(1), ds2465(ds2465),
      ds28e15(MaximInterface::mbed::Sleep::instance(), ds2465, &skipRom),
      ds7505(i2c, ds7505_i2c_addr), max44009(i2c, max44009_i2c_addr) {
  if (!rngInitialized) {
    initializeRng();
    rngInitialized = true;
  }
}

bool SensorNode::initializeSensors() {
  return (max44009.read_current_lux(initialLux_) == MAX44009::Success);
}

bool SensorNode::setSecret() {
  // Create constant partial secret
  DS28E15::Scratchpad scratchpad;
  scratchpad.fill(uint8_t(defaultPaddingByte));
  // Calculate secret
  const Sha256::SlaveSecretData data = DS28E15::createSlaveSecretData(
      DS28E15::Page(), authData.pageNum, scratchpad, romId_, manId);
  return !ds2465.computeSlaveSecretWithSwap(data, 0, DS2465::FullPage);
}

bool SensorNode::checkProvisioned(bool & provisioned) {
  DS28E15::BlockProtection protectionStatus;
  bool result;

  result = !ds28e15.readBlockProtection(0, protectionStatus);
  if (result) {
    if (!protectionStatus.noProtection()) {
      result = !ds28e15.readSegment(authData.pageNum, authData.segmentNum,
                                    authData.segment);
      if (result)
        provisioned = true;
    } else {
      provisioned = false;
    }
  }
  return result;
}

bool SensorNode::checkAuthentic(unsigned int userEntropy) {
  DS28E15::Scratchpad challenge;
  DS28E15::Page pageData;

  // Read page data
  if (ds28e15.readPage(authData.pageNum, pageData))
    return false;

  // Create random challenge
  // Use hardare RNG on MAX32600
#ifdef TARGET_MAX32600
  MXC_TPU->prng_user_entropy = userEntropy;
#else
  std::srand(userEntropy);
#endif
  for (size_t i = 0; i < challenge.size(); i++) {
#ifdef TARGET_MAX32600
    challenge[i] = MXC_TPU->prng_rnd_num;
#else
    challenge[i] = std::rand();
#endif
  }

  // Write challenge to scratchpad
  if (ds28e15.writeScratchpad(challenge))
    return false;
  // Have device compute MAC
  Sha256::Hash nodeMac;
  if (ds28e15.computeReadPageMac(0, false, nodeMac))
    return false;
  // Compute expected MAC
  const Sha256::AuthMacData controllerMacData = DS28E15::createAuthMacData(
      pageData, authData.pageNum, challenge, romId_, manId);
  Sha256::Hash controllerMac;
  if (ds2465.computeAuthMac(controllerMacData, controllerMac))
    return false;
  // Check if authentic
  return (nodeMac == controllerMac);
}

bool SensorNode::readSensorData(SensorData & sensorData) {
  bool result;
  int8_t temp;

  // Read temperature sensor
  result = (ds7505.read_current_temp(temp) == DS7505::Success);

  if (result) {
    sensorData.temp = temp;

    // Read light sensor
    double currentLux;
    result = (max44009.read_current_lux(currentLux) == MAX44009::Success);
    if (result) {
      // Convert lux to remaining filter life
      sensorData.filterLife =
          static_cast<uint8_t>((currentLux / initialLux_) * 100);
    }
  }

  return result;
}

bool SensorNode::checkAndWriteAuthData(SensorData & sensorData) {
  bool result = true;

  if (sensorData.filterLife > authData.filterLife()) {
    sensorData.filterLife = authData.filterLife();
  } else if (sensorData.filterLife < authData.filterLife()) {
    AuthData newAuthData = authData;
    newAuthData.filterLife() = sensorData.filterLife;
    const Sha256::WriteMacData macData = DS28E15::createSegmentWriteMacData(
        authData.pageNum, authData.segmentNum, newAuthData.segment,
        authData.segment, romId_, manId);
    Sha256::Hash mac;
    result = !ds2465.computeWriteMac(macData, mac);
    if (result)
      result = !ds28e15.writeAuthSegment(authData.pageNum, authData.segmentNum,
                                         newAuthData.segment, mac);
    if (result)
      authData = newAuthData;
  }

  return result;
}

SensorNode::State SensorNode::detect(unsigned int userEntropy) {
  bool provisioned;

  ds2465.setSpeed(DS2465::OverdriveSpeed);

  if (readRom(ds2465, romId_))
    return UnableToCommunicate;

  if (!checkProvisioned(provisioned))
    return UnableToCommunicate;

  if (!provisioned)
    return NotProvisioned;

  if (!setSecret())
    return UnableToCommunicate;

  if (!checkAuthentic(userEntropy))
    return NotAuthentic;

  if (!initializeSensors())
    return UnableToCommunicate;

  return Authentic;
}

SensorNode::State
SensorNode::authenticatedReadSensorData(unsigned int userEntropy,
                                        SensorData & sensorData) {
  ds2465.setSpeed(DS2465::OverdriveSpeed);

  if (!setSecret())
    return UnableToCommunicate;

  if (!checkAuthentic(userEntropy))
    return NotAuthentic;

  if (!readSensorData(sensorData))
    return UnableToCommunicate;

  if (!checkAndWriteAuthData(sensorData))
    return NotAuthentic;

  return Authentic;
}