/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef DISPLAY_HPP
#define DISPLAY_HPP

#include <stdint.h>
#include <stddef.h>
#include <string>

namespace mbed { class I2C; }

/// Interface to the Newhaven Display NHD-C0220BiZ-FS(RGB)-FBW-3VM LCD module
/// and MAX7306 PWM LED driver for backlight color selection.
class Display {
public:
  /// Display line referenced from top
  enum Line { FirstLine = 0, SecondLine };

  /// 24-bit RGB color for the backlight.
  struct Color {
    uint8_t R, G, B;
  };

  /// Length in character os a display line.
  static const size_t lineLength = 20;

  /// @param I2C_interface A configured I2C interface to use for communication.
  /// @param LCD_I2C_address LCD module bus address in mbed format.
  /// @param LED_driver_I2C_addr PWM LED driver (MAX7306) bus address in mbed format.
  Display(mbed::I2C & I2C_intf, uint8_t LCD_I2C_addr,
          uint8_t LED_driver_I2C_addr);

  /// Initialize display components.
  void initialize();

  /// Clear all display lines.
  void clearDisplay();

  /// Clear a specific display line.
  void clearLine(Line line);

  /// Write a single character to the display at the current cursor position.
  void writeCharacter(uint8_t character);

  /// Write text to the display at the current cursor position.
  void writeText(const std::string & text);

  /// Set cursor to a certain line and zero-index position within the line.
  void setCursorPosition(Line line, size_t position = 0);

  /// Writes text to the display starting at the beginning of the line.
  void writeLine(const std::string & text, Line line);

  /// Writes text to the display starting at the beginning of the line and
  /// clears the remainder of the line.
  void writeCompleteLine(const std::string & text, Line line);

  /// Writes a message to the display with text wrapping allowed at spaces.
  void writeMessage(const std::string & text);

  /// Set the display backlight to a certain color.
  void setBackLightColor(const Color & color);

private:
  mbed::I2C & m_I2C_intf;
  uint8_t m_LCD_I2C_addr, m_LED_driver_I2C_addr;

  void initializeLCD();
  void initializeLED_Driver();
};

#endif
