/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#include "Factory.hpp"
#include "SensorNode.hpp"
#include "common.hpp"
#include "Masters/DS2465/DS2465.h"
#include "Slaves/Authenticators/DS28E15_22_25/DS28E15_22_25.h"
#include "wait_api.h"

using namespace OneWire;

const uint8_t Factory::masterSecret[] = { 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x21,
                                              0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x31, 0x32 };
const uint8_t Factory::invalidMasterSecret[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                                     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

bool Factory::provision(DS2465 & ds2465)
{
  bool result = (ds2465.setMasterSecret(masterSecret) == ISha256MacCoproc::Success);
  if (result)
  {
    SensorNode::AuthData authData;
    DS28E15_22_25::Page pageData;
    std::memset(pageData, SensorNode::defaultPaddingByte, pageData.length);
    std::memcpy(pageData, authData.segment, authData.segment.length);
    result = (ds2465.writeScratchpad(pageData, pageData.length) == OneWireMaster::Success);
  }
  if (result)
  {
    result = (ds2465.copyScratchpadToPage(0) == OneWireMaster::Success);
    if (result)
      wait_ms(DS2465::eepromPageWriteDelayMs);
  }
  
  return result;
}

bool Factory::provision(SensorNode & sensorNode, bool validSecret)
{
  const int blockNum = sensorNode.authData.pageNum / 2;
  const DS28E15_22_25::BlockProtection desiredProtection(false, false, false, true, blockNum); // Authentication Protection only
  
  // Reset to starting defaults
  sensorNode.authData.reset();
  
  // Read current protection status
  DS28E15_22_25::BlockProtection protectionStatus;
  bool result;
  // Read block protections
  result = (sensorNode.ds28e15.readBlockProtection(blockNum, protectionStatus) == OneWireSlave::Success);
  // Check if invalid protections are set
  if (result)
    result = ((protectionStatus.statusByte() & ~(desiredProtection.statusByte())) == 0x00);
  // Load secret into scratchpad
  if (result)
    result = (sensorNode.ds28e15.writeScratchpad(validSecret ? masterSecret : invalidMasterSecret) == OneWireSlave::Success);
  // Load master secret from scratchpad without locking
  if (result)
    result = (sensorNode.ds28e15.loadSecret(false) == OneWireSlave::Success);
  
  // Setup is complete if not using a valid secret
  if (!validSecret)
    return result;
  
  // Create constant partial secret
  DS28E15_22_25::Scratchpad partialSecret;
  DS28E15_22_25::Page pageData;
  std::memset(partialSecret, SensorNode::defaultPaddingByte, partialSecret.length);
  
  // Read page data
  if (result)
    result = (sensorNode.ds28e15.readPage(sensorNode.authData.pageNum, pageData, false) == OneWireSlave::Success);
  // Load partial secret into scratchpad
  if (result)
    result = (sensorNode.ds28e15.writeScratchpad(partialSecret) == OneWireSlave::Success);
  // Compute secret
  if (result)
    result = (sensorNode.ds28e15.computeSecret(sensorNode.authData.pageNum, false) == OneWireSlave::Success);
  // Configure slave secret on DS2465
  if (result)
    result = (DS28E15_22_25::computeNextSecret(sensorNode.ds2465, pageData, sensorNode.authData.pageNum, partialSecret, sensorNode.ds28e15.romId(), sensorNode.ds28e15.manId()) == ISha256MacCoproc::Success);
  // Enable authentication protection if not set
  if (result && (protectionStatus != desiredProtection))
    result = (sensorNode.ds28e15.writeAuthBlockProtection(sensorNode.ds2465, desiredProtection, protectionStatus) == OneWireSlave::Success);
  // Write initial filter life and set all other segments to default value
  if (result)
  {
    DS28E15_22_25::Segment blankSegment;
    std::memset(blankSegment, SensorNode::defaultPaddingByte, blankSegment.length);
    for (size_t i = 0; i < DS28E15_22_25::segmentsPerPage; i++)
    {
      result = (sensorNode.ds28e15.writeAuthSegment(sensorNode.ds2465, sensorNode.authData.pageNum, i,
                                                          ((i == sensorNode.authData.segmentNum) ? sensorNode.authData.segment : blankSegment),
                                                          pageData.toSegment(i), false) == OneWireSlave::Success);
      
      if (!result)
        break;
    }
  }
  
  // Reload secret with known page values
  // Load master secret into scratchpad
  if (result)
    result = (sensorNode.ds28e15.writeScratchpad(masterSecret) == OneWireSlave::Success);
  // Load master secret
  if (result)
    result = (sensorNode.ds28e15.loadSecret(false) == OneWireSlave::Success);
  // Read page data
  if (result)
    result = (sensorNode.ds28e15.readPage(sensorNode.authData.pageNum, pageData, false) == OneWireSlave::Success);
  // Write partial secret to scratchpad
  if (result)
    result = (sensorNode.ds28e15.writeScratchpad(partialSecret) == OneWireSlave::Success);
  // Compute secret
  if (result)
    result = (sensorNode.ds28e15.computeSecret(sensorNode.authData.pageNum, false) == OneWireSlave::Success);
  // Configure slave secret on DS2465
  if (result)
    result = (DS28E15_22_25::computeNextSecret(sensorNode.ds2465, pageData, sensorNode.authData.pageNum, partialSecret, sensorNode.ds28e15.romId(), sensorNode.ds28e15.manId()) == ISha256MacCoproc::Success);

  return result;
}