/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#include <sstream>
#include "Display.hpp"
#include "I2C.h"
#include "wait_api.h"

//LCD Commands
//If the RS bit is set to logic 1, these display bytes are stored in the display RAM at the address specified by the data pointer. The data pointer is
//automatically updated and the data is directed to the intended ST7036i device. If the RS bit of the last control byte is set to
//logic 0, these command bytes will be decoded and the setting of the device will be changed according to the received commands.
enum LCD_Commands
{
  ControlByte = 0x00,	        //Only one control byte will be sent. Only a stream of data bytes is allowed to follow.
  ControlByte_RS_Set = 0x40,  	//Only one control byte will be sent with the RS bit set. Only a stream of data bytes is allowed to follow.
  ControlBytes = 0x80,          //Another control byte will follow, unless an I2C Stop condition is received.
  ControlBytes_RS_Set = 0xC0, 	//RS Set and another control byte will follow, unless an I2C Stop condition is received.
};

//LCD Instructions
enum LCD_Instructions
{
  ClearDisplay = 0x01,
  Display_OFF = 0x08,     	//Display off
  Display_ON = 0x0C,     	        //Display on, cursor off, cursor position off
  ReturnHome = 0x02,
  SetDdramAddress = 0x80
};

// LED Driver Port Registers
// Initial port state 0x80
enum LED_Driver_Ports
{
  P1 = 0x01,
  P2 = 0x02,  // Blue LED
  P3 = 0x03,  // Green LED
  P4 = 0x04   // Red LED
};

// Convert a byte color value into the representation used by the MAX7306 PWM registers
static uint8_t convertColorToPwmRegVal(uint8_t color)
{
  const uint8_t staticOffRegVal = 0x80; // LED is static off by setting to input
  const uint8_t staticOnRegVal = 0x00; // LED is static on
  const uint8_t minOnRegVal = 0x01; // LED on for minimum duty cycle
  
  uint8_t regVal;
  if (color == 0x00) // Use static off for no color
  {
    regVal = staticOffRegVal;
  }
  else if (color == 0xFF) // Use static on for full color
  {
    regVal = staticOnRegVal;
  }
  else // Use standard PWN for all other values
  {
    // The 3 least significant bits cannot be rendered with the MAX7306
    regVal = color >> 3;
    if (regVal == staticOnRegVal)
      regVal = minOnRegVal;
  }
  return regVal;
}

Display::Display(mbed::I2C & I2C_intf, uint8_t LCD_I2C_addr, uint8_t LED_driver_I2C_addr)
  : m_I2C_intf(I2C_intf), m_LCD_I2C_addr(LCD_I2C_addr), m_LED_driver_I2C_addr(LED_driver_I2C_addr)
{
  
}

void Display::initialize(void)
{
  initializeLCD();
  initializeLED_Driver();
}

void Display::initializeLED_Driver(void)
{
  const uint8_t Configuration26 = 0x26;  //intial port state 0xEC
  const uint8_t Configuration27 = 0x27;  //intial port state 0x8F
  
  //Intial mode
  //write to Configuration Register 0x26
  m_I2C_intf.start();
  m_I2C_intf.write(m_LED_driver_I2C_addr);
  m_I2C_intf.write(Configuration26);
  //RST does  reset PWM/blink counters, RST resets registers to power-on-reset state
  m_I2C_intf.write(0x1F); 
  m_I2C_intf.stop();
  
  //Write to Configuration Register 0x27
  m_I2C_intf.start();
  m_I2C_intf.write(m_LED_driver_I2C_addr);
  m_I2C_intf.write(Configuration27);
  //Enable bus time out, set P1,P2,P3 to be controlled by their registers (0x01,0x02,0x03)
  m_I2C_intf.write(0x0E); 
  m_I2C_intf.stop();
}

void Display::setBackLightColor(const Color & color)
{  
  // Red
  m_I2C_intf.start();
  m_I2C_intf.write(m_LED_driver_I2C_addr);
  m_I2C_intf.write(P4);
  m_I2C_intf.write(convertColorToPwmRegVal(color.R)); 
  m_I2C_intf.stop();
   
  // Green
  m_I2C_intf.start();
  m_I2C_intf.write(m_LED_driver_I2C_addr);
  m_I2C_intf.write(P3);
  m_I2C_intf.write(convertColorToPwmRegVal(color.G)); 
  m_I2C_intf.stop();
  
  // Blue
  m_I2C_intf.start();
  m_I2C_intf.write(m_LED_driver_I2C_addr);
  m_I2C_intf.write(P2);
  m_I2C_intf.write(convertColorToPwmRegVal(color.B)); 
  m_I2C_intf.stop();
}

void Display::clearLine(Line line)
{
  writeCompleteLine("", line);
  setCursorPosition(line);
}

void Display::clearDisplay(void)
{
  m_I2C_intf.start();
  m_I2C_intf.write(m_LCD_I2C_addr);
  m_I2C_intf.write(ControlByte);     //No more control bytes will be sent
  m_I2C_intf.write(ClearDisplay);
  m_I2C_intf.stop();
}

void Display::initializeLCD(void)
{
  m_I2C_intf.start();
  m_I2C_intf.write(m_LCD_I2C_addr);
  m_I2C_intf.write(ControlByte);    //No more control bytes will be sent
  m_I2C_intf.write(0x38);               //Function Set IS[2:1] = 0,0 (&h38 = Single height font, 0x3C = double height font)
  m_I2C_intf.write(0x39);               //Function Set IS[2:1] = (0,1)
  //When IS[2:1]=(0,0): normal instruction be selected(refer instruction table 0)
  //When IS[2:1]=(0,1): extension instruction be selected(refer instruction table 1 )
  //When IS[2:1]=(1,0): extension instruction be selected(refer instruction table 2 )
  m_I2C_intf.write(0x14);               //BIAS SET
  m_I2C_intf.write(0x70);               //CONTRAST (was 0x78)
  m_I2C_intf.write(0x5E);               //POWER/ICON CONTROL/CONTRAST (upper two bits)
  m_I2C_intf.write(0x6D);               //FOLLOWER CONTROL
  m_I2C_intf.stop();
  wait_ms(200);                         //Wait for power stable
  m_I2C_intf.start();
  m_I2C_intf.write(m_LCD_I2C_addr);
  m_I2C_intf.write(ControlByte);        //No more control bytes will be sent
  m_I2C_intf.write(Display_ON);         //Display on, cursor on, cursor position on
  m_I2C_intf.write(ClearDisplay);       //Clear Display
  m_I2C_intf.write(0x06);               //ENTRY MODE
  m_I2C_intf.stop();
}

void Display::writeCharacter(uint8_t character)
{
  m_I2C_intf.start();
  m_I2C_intf.write(m_LCD_I2C_addr);
  m_I2C_intf.write(ControlByte_RS_Set);	//No more control bytes will be sent
  m_I2C_intf.write(character);	        //Display on, cursor on, cursor position on
  m_I2C_intf.stop();	
}

void Display::writeText(const std::string & text)
{
  const char RETURN_CHAR = 0x16;
  
  size_t length = text.length();
  if (length > lineLength)
    length = lineLength;
  
  //Write to LCD
  m_I2C_intf.start();
  m_I2C_intf.write(m_LCD_I2C_addr);
  m_I2C_intf.write(ControlByte_RS_Set);
  
  for(size_t i = 0; i < length; i++)
  {
    if(text[i] != RETURN_CHAR)
      m_I2C_intf.write(text[i]);
  }
  
  m_I2C_intf.stop();
}

void Display::setCursorPosition(Line line, size_t position)
{  
  if (position > (lineLength - 1)) // Set to last line character for values outside the upper bound
    position = (lineLength - 1);

  m_I2C_intf.start();
  m_I2C_intf.write(m_LCD_I2C_addr);
  m_I2C_intf.write(ControlByte); // No more control bytes will be sent
  if(line == SecondLine) // Offset for second line
    position += 0x40;
  m_I2C_intf.write(SetDdramAddress | position);
  m_I2C_intf.stop();
}

void Display::writeLine(const std::string & text, Line line)
{  
  setCursorPosition(line);
  writeText(text);
}

void Display::writeCompleteLine(const std::string & text, Line line)
{
  // Add padding to user's string
  std::string writeText(text);
  if (writeText.length() < lineLength)
    writeText.append(lineLength - writeText.length(), ' ');
  
  writeLine(writeText, line);
}

void Display::writeMessage(const std::string & message)
{
  if (message.length() > lineLength)
  {
    // Find split point
    std::istringstream messageStream(message);
    std::string word;
    size_t splitIndex = 0;
    do
    {
      if (word.length() > 0)
        splitIndex += (word.length() + 1);
      std::getline(messageStream, word, ' ');
    } while ((splitIndex + word.length()) <= lineLength);
    if (splitIndex == 0) // First word is too long
    {
      writeCompleteLine(message.substr(0, lineLength), FirstLine);
      writeCompleteLine(message.substr(lineLength), SecondLine);
    }
    else
    {
      writeCompleteLine(message.substr(0, splitIndex - 1), FirstLine);
      writeCompleteLine(message.substr(splitIndex), SecondLine);
    }
  }
  else
  {
    writeCompleteLine(message, FirstLine);
    writeCompleteLine("", SecondLine);
  }
}
