/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#include "DS7505.hpp"
#include "I2C.h"
#include "wait_api.h"

#define I2C_WRITE 0
#define I2C_READ 1

static const int I2C_WRITE_OK = 0;
static const uint8_t DS7505_Config_SD_Bit = 0x01; // Enable shutdown mode

DS7505::DS7505(mbed::I2C & I2C_interface, uint8_t I2C_address)
  : m_current_config(Config_9b_Res, true), m_I2C_interface(I2C_interface), m_I2C_address(I2C_address)
{
  
}

uint8_t DS7505::get_measure_delay_ms(Config_Resolution resolution)
{
  uint8_t measure_delay_ms;
  
  switch (resolution)
  {
    case Config_9b_Res:
      measure_delay_ms = 25;
      break;
    case Config_10b_Res:
      measure_delay_ms = 50;
      break;
    case Config_11b_Res:
      measure_delay_ms = 100;
      break;
    case Config_12b_Res:
      measure_delay_ms = 200;
      break;
    default:
      measure_delay_ms = 0;
      break;
  }
  
  return measure_delay_ms;
}

bool DS7505::read_temp_sensor_data(uint16_t & sensor_data) const
{
  bool result;
  uint8_t upperByte, lowerByte;
  int sub_res;
  
  sensor_data = 0;
  m_I2C_interface.start();
  sub_res = m_I2C_interface.write(m_I2C_address | I2C_READ);
  if (sub_res == I2C_WRITE_OK)
  {
    upperByte = m_I2C_interface.read(mbed::I2C::ACK);
    lowerByte = m_I2C_interface.read(mbed::I2C::NoACK);
  }
  m_I2C_interface.stop();
  if (sub_res == I2C_WRITE_OK)
  {
    sensor_data = ((((uint16_t)upperByte) << 8) | lowerByte);
    result = true;
  }
  else
  {
    // Handle hardware malfunction
    result = false;
  }
  
  return result;
}

bool DS7505::set_register_pointer(Register pointer_reg) const
{
  int res;
  
  m_I2C_interface.start();
  res = m_I2C_interface.write(m_I2C_address | I2C_WRITE);
  if (res == I2C_WRITE_OK)
  {
    res = m_I2C_interface.write(pointer_reg);
  }
  m_I2C_interface.stop();
  
  return (res == I2C_WRITE_OK);
}

bool DS7505::write_register(Register write_reg, uint8_t write_val) const
{
  bool res;
  
  m_I2C_interface.start();
  res = m_I2C_interface.write(m_I2C_address | I2C_WRITE);
  if (res == I2C_WRITE_OK)
  {
    res = m_I2C_interface.write(write_reg);
    if (res == I2C_WRITE_OK)
      res = m_I2C_interface.write(write_val);
  }
  m_I2C_interface.stop();
  
  return (res == I2C_WRITE_OK);
}

bool DS7505::write_current_config() const
{
  uint8_t DS7505_Config_Val = m_current_config.resolution;
  if (m_current_config.enable_shutdown_mode)
    DS7505_Config_Val |= DS7505_Config_SD_Bit;
  return write_register(Configuration_Reg, DS7505_Config_Val);
}

DS7505::Result DS7505::set_resolution(uint8_t resolution)
{   
  switch (resolution)
  {
    case 1:
      m_current_config.resolution = Config_9b_Res;
      break;
    case 2:
      m_current_config.resolution = Config_10b_Res;
      break;
    case 3:
      m_current_config.resolution = Config_11b_Res;
      break;
    case 4:
      m_current_config.resolution = Config_12b_Res;
      break;
    default:
      return Out_of_Range;
  }
  
  // Write DS7505 configuration
  if (!write_current_config())
  {
    // Handle hardware malfunction
    return Hardware_Failure;
  }
  
  // Set pointer to temperature register
  if (!set_register_pointer(Temperature_Reg))
  {
    // Handle hardware malfunction
    return Hardware_Failure;
  }
  
  return Success;
}

DS7505::Result DS7505::read_temp_sensor(uint16_t & sensor_data) const
{
  bool res;
  
  if (m_current_config.enable_shutdown_mode)
  {
    // Disable shutdown mode
    m_current_config.enable_shutdown_mode = false;
    res = write_current_config();
    if (!res)
      return Hardware_Failure;
  
    // DS7505 measures temperature
  
    // Enable shutdown mode
    m_current_config.enable_shutdown_mode = true;
    res = write_current_config();
    if (!res)
      return Hardware_Failure;
    
    // Set pointer to temperature register
    res = set_register_pointer(Temperature_Reg);
    if (!res)
      return Hardware_Failure;
    
    // Sleep for maximum time needed for sample
    wait_ms(get_measure_delay_ms(m_current_config.resolution));
  }
  // else: shutdown mode disabled
  //    DS7505 is constantly measuring temperature
  
  // Read temperature from sensor
  if (!read_temp_sensor_data(sensor_data))
  {
    return Hardware_Failure;
  }
  
  return Success;
}

DS7505::Result DS7505::read_current_temp(int16_t & temperature) const
{
  uint16_t sensor_data;
  Result result;
  
  result = read_temp_sensor(sensor_data);
  if (result == Success)
  {
    // Convert temperature to have an exponent of 10^-2
    temperature = ((int8_t)(sensor_data >> 8)) * 100;
    if (sensor_data & 0x0080)
      temperature += 50; // 0.5
    if (sensor_data & 0x0040)
      temperature += 25; // 0.25
    if (sensor_data & 0x0020)
      temperature += 13; // 0.125
    if (sensor_data & 0x0010)
      temperature += 6; // 0.0625
  }
  return result;
}

DS7505::Result DS7505::read_current_temp(double & temperature) const
{
  uint16_t sensor_data;
  Result result;
  
  result = read_temp_sensor(sensor_data);
  if (result == Success)
  {
    // Convert sensor data to floating-point temperature
    temperature = ((int8_t)(sensor_data >> 8));
    if (sensor_data & 0x0080)
      temperature += 0.5;
    if (sensor_data & 0x0040)
      temperature += 0.25;
    if (sensor_data & 0x0020)
      temperature += 0.125;
    if (sensor_data & 0x0010)
      temperature += 0.0625;
  }
  return result;
}

DS7505::Result DS7505::read_current_temp(int8_t & temperature) const
{
  uint16_t sensor_data;
  Result result;
  
  result = read_temp_sensor(sensor_data);
  if (result == Success)
  {
    // Convert sensor data to integer temperature
    temperature = ((int8_t)(sensor_data >> 8));
  }
  return result;
}