/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#include "SensorNode.hpp"
#include "common.hpp"
#include "Masters/DS2465/DS2465.h"
#include "RomId/RomCommands.h"
#include "I2C.h"

#ifdef TARGET_MAX32600
#include "max32600.h"
#include "clkman_regs.h"
#include "tpu_regs.h"
#else
#include <cstdlib>
#endif

using namespace OneWire;
using namespace OneWire::RomCommands;

bool SensorNode::rngInitialized = false;

void SensorNode::initializeRng()
{
#ifdef TARGET_MAX32600
  MXC_CLKMAN->clk_config |= (MXC_F_CLKMAN_CLK_CONFIG_CRYPTO_ENABLE | MXC_F_CLKMAN_CLK_CONFIG_CRYPTO_RESET_N); // Enable crypto oscillator
  while ((MXC_CLKMAN->intfl & MXC_F_CLKMAN_INTFL_CRYPTO_STABLE) != MXC_F_CLKMAN_INTFL_CRYPTO_STABLE) ; // Wait for crypto oscillator stability
  MXC_CLKMAN->clk_ctrl |= MXC_F_CLKMAN_CLK_CTRL_CRYPTO_GATE_N; // Disable crypto clock gating
  MXC_CLKMAN->crypt_clk_ctrl_2_prng = MXC_CLKMAN->clk_ctrl_10_prng = 1; // Set PRNG clock to crypto clock
  MXC_CLKMAN->clk_gate_ctrl2 |=  (1 << MXC_F_CLKMAN_CLK_GATE_CTRL2_TPU_CLK_GATER_POS); // Use dynamic clock gating
#endif
}

SensorNode::SensorNode(mbed::I2C & i2c, uint8_t ds7505_i2c_addr, uint8_t max44009_i2c_addr, DS2465 & ds2465)
  : m_initialLux(1),ds2465(ds2465), selector(ds2465), ds28e15(selector), ds7505(i2c, ds7505_i2c_addr), max44009(i2c, max44009_i2c_addr)
{
  if (!rngInitialized)
  {
    initializeRng();
    rngInitialized = true;
  }
}

bool SensorNode::initializeSensors()
{
  return (max44009.read_current_lux(m_initialLux) == MAX44009::Success);
}

bool SensorNode::setSecret()
{
  DS28E15_22_25::Scratchpad scratchpad;
  DS28E15_22_25::Page pageData;
  
  // Create constant partial secret
  std::memset(scratchpad, defaultPaddingByte, scratchpad.length);
  // Read page data
  bool result = (ds2465.readMemory(DS2465::UserMemoryPage0, pageData, pageData.length, false) == OneWireMaster::Success);
  // Calculate secret
  if (result)
  {
    result = (DS28E15_22_25::computeNextSecret(ds2465, pageData, authData.pageNum, scratchpad, ds28e15.romId(), ds28e15.manId()) == ISha256MacCoproc::Success);
  }
  return result;
}

bool SensorNode::checkProvisioned(bool & provisioned)
{
  DS28E15_22_25::BlockProtection protectionStatus;
  bool result;
  
  
  result = (ds28e15.readBlockProtection(0, protectionStatus) == OneWireSlave::Success);
  if (result)
  {
    if (!protectionStatus.noProtection())
    {
      result = (ds28e15.readSegment(authData.pageNum, authData.segmentNum, authData.segment) == OneWireSlave::Success);
      if (result)
        provisioned = true;
    }
    else
    {
      provisioned = false;
    }
  }
  return result;
}

bool SensorNode::checkAuthentic(unsigned int userEntropy)
{
  DS28E15_22_25::Scratchpad challenge;
  DS28E15_22_25::Page pageData;
  
  // Read page data
  if (ds28e15.readPage(authData.pageNum, pageData, false) != OneWireSlave::Success)
    return false;
  
  // Create random challenge
  // Use hardare RNG on MAX32600
#ifdef TARGET_MAX32600 
  MXC_TPU->prng_user_entropy = userEntropy;
#else
  std::srand(userEntropy);
#endif
  for (size_t i = 0; i < challenge.length; i++)
  {
#ifdef TARGET_MAX32600
    challenge[i] = MXC_TPU->prng_rnd_num;
#else
    challenge[i] = std::rand(); 
#endif
  }

  // Write challenge to scratchpad
  if (ds28e15.writeScratchpad(challenge) != OneWireSlave::Success)
    return false;
  // Have device compute MAC
  DS28E15_22_25::Mac nodeMac;
  if (ds28e15.computeReadPageMac(0, false, nodeMac) != OneWireSlave::Success)
    return false;
  // Compute expected MAC
  DS28E15_22_25::Mac controllerMac;
  if (DS28E15_22_25::computeAuthMac(ds2465, pageData, authData.pageNum, challenge, ds28e15.romId(), ds28e15.manId(), controllerMac) != ISha256MacCoproc::Success)
    return false;
  // Check if authentic
  return (nodeMac == controllerMac);
}

bool SensorNode::readSensorData(SensorData & sensorData)
{
  bool result;
  int8_t temp;
  
  // Read temperature sensor
  result = (ds7505.read_current_temp(temp) == DS7505::Success);
  
  if (result)
  {
    sensorData.temp = temp;
    
    // Read light sensor
    double currentLux;
    result = (max44009.read_current_lux(currentLux) == MAX44009::Success);
    if (result)
    {
      // Convert lux to remaining filter life
      sensorData.filterLife = (unsigned int)((currentLux / m_initialLux) * 100);
    }
  }
  
  return result;
}

bool SensorNode::checkAndWriteAuthData(SensorData & sensorData)
{
  bool result = true;
  
  if (sensorData.filterLife > authData.filterLife)
  {
    sensorData.filterLife = authData.filterLife;
  }
  else if (sensorData.filterLife < authData.filterLife)
  {
    AuthData oldAuthData(authData);
    authData.filterLife = sensorData.filterLife;
    // Write new filter life to DS28E15
    result = (ds28e15.writeAuthSegment(ds2465, authData.pageNum, authData.segmentNum, authData.segment, oldAuthData.segment, false) == OneWireSlave::Success);
  }
  
  return result;
}

SensorNode::State SensorNode::detect(unsigned int userEntropy)
{
  bool provisioned;
  
  ds2465.OWSetSpeed(DS2465::OverdriveSpeed);
  
  RomId romId;
  if (OWReadRom(ds2465, romId) != OneWireMaster::Success)
    return UnableToCommunicate;
  ds28e15.setRomId(romId);
  
  if (!checkProvisioned(provisioned))
    return UnableToCommunicate;
  
  if (!provisioned)
      return NotProvisioned;
  
  if (!setSecret())
    return UnableToCommunicate;
    
  if (!checkAuthentic(userEntropy))
    return NotAuthentic;
  
  if (!initializeSensors())
    return UnableToCommunicate;
  
  return Authentic;
}

SensorNode::State SensorNode::authenticatedReadSensorData(unsigned int userEntropy, SensorData & sensorData)
{
  ds2465.OWSetSpeed(DS2465::OverdriveSpeed);
  
  if (!setSecret())
    return UnableToCommunicate;
  
  if (!checkAuthentic(userEntropy))
    return NotAuthentic;
  
  if (!readSensorData(sensorData))
    return UnableToCommunicate;
  
  if (!checkAndWriteAuthData(sensorData))
    return NotAuthentic;
  
  return Authentic;
}