/*******************************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#ifndef _ESP8266_HPP
#define _ESP8266_HPP

#include <string>
#include <sstream>

#include "PinNames.h"
#include "Serial.h"
#include "DigitalOut.h"
#include "CircularBuffer.h"

namespace mbed { class Serial; }

/// Interface to the ESP8266 Wi-Fi module.
class ESP8266
{
public:
  /// Result of sending an AT command.
  enum CmdResult
  {
    AT_OK = 1,
    AT_FAIL = 0,
    AT_ERROR = -1,
    HardwareError = -2,
    TimeoutError = -3
  };
  
  /// ESP8266 Wi-Fi mode.
  enum WifiMode
  {
    station_mode = 1,
    softAP_mode = 2,
    softAP_station_mode = 3
  };
  
  /// Connection type.
  enum ConnType
  {
    TCP,
    UDP
  };
  
  /// Recovery time between Send Data operation as specified by datasheet.
  static const unsigned int sendDataRecoveryTimeMs = 1000;
  
  /// Builds command strings for the ESP8266 with proper formatting.
  class CmdBuilder
  {
  public:
    /// @param cmd Command of the format "AT+[cmd]".
    CmdBuilder(const std::string & cmd = "");
    
    /// Clear all arguments.
    /// @param cmd Command of the format "AT+[cmd]".
    void clear(const std::string & cmd);
    
    /// Append an argument using the default string conversion for that type.
    /// @param arg Argument to append to the command.
    template <typename T> void addRawArgument(const T & arg)
    {
      cmdStream << ((numArgs == 0) ? "=" : ",") << arg;
      numArgs++;
    }
    
    /// Append a string argument with proper quoting.
    /// @param arg Argument to append to the command.
    void addStringArgument(const std::string & arg);
    
    /// Create a string suitable for use with sendCommand().
    /// @returns The formatted command string.
    std::string str() const;
    
  private:
    int numArgs;
    std::ostringstream cmdStream;
  };
  
  /// @{
  /// Default instance support for use with mbed Sockets.
  static void setDefaultInstance(ESP8266 * const instance);
  static ESP8266** getDefaultInstance();
  /// @}
  
  /// @param tx Transmit pin from mbed to ESP8266.
  /// @param rx Receive pin from ESP8266 to mbed.
  /// @param rst Reset pin on ESP8266.
  /// @param CH_PD Power-down pin on ESP8266.
  /// @param baud Baud rate that the ESP8266 is using.
  /// @param debugMsgIntf Optional serial interface for debugging messages.
  ESP8266(const PinName tx, const PinName rx, const PinName rst, const PinName CH_PD, const int baud, mbed::Serial * debugMsgIntf = NULL);
  ~ESP8266();
  
  /// Reset the ESP8266 via the hardware reset pin.
  void reset();
  
  // Update the baud rate for the ESP8266.
  void setBaud(int baud) { AT_intf.baud(baud); }
  
  /// @{
  /// Control if the ESP8266 is powered via the hardware power-down pin.
  bool powered() const;
  void setPowered(bool powered);
  /// @}
  
  /// Perform a self-test on the ESP8266.
  CmdResult performSelfTest();
  
  /// Set the current Wi-Fi mode.
  CmdResult setCurrentWifiMode(const WifiMode mode);
  
  /// Join a Wi-Fi access point.
  /// @param ssid Network SSID to connect to.
  /// @param pwd Network password.
  /// @param bssid Optional network BSSID.
  CmdResult joinCurrentAccessPoint(const std::string & ssid, const std::string & pwd, const std::string & bssid = "");
  
  /// Quit the current access point.
  CmdResult quitAccessPoint();
  
  /// Set the maximum WiFi tranmission power.
  /// @param power_dBm Power in dBm valid from 0 to 20.5 in 0.25 dBm increments.
  CmdResult setMaxRFTXPower(const float power_dBm);
  
  /// Ping a host via the current access point.
  /// @param IP IP address or resolvable hostname.
  CmdResult ping(const std::string & IP);
  
  /// Open a connection to a host via the current access point.
  /// @param type TCP or UPD connection.
  /// @param remoteIP IP address or resolvable hostname to connect to.
  /// @param remotePort Port on the host to connect to.
  CmdResult openConnection(const ConnType type, const std::string & remoteIP, const unsigned int remotePort);
  
  /// Close the connection to the current host.
  CmdResult closeConnection();
  
  /// Send data to the currently connected host.
  /// @param data May be in text or binary form.
  CmdResult sendData(const std::string & data);
  
  /// Send an AT command to the ESP8266.
  /// @param cmd Formatted command to send.
  CmdResult sendCommand(const CmdBuilder & cmd);
  
  /// Check if received IP data is available in the buffer.
  /// @note Allow some processing delay to happen between calls to this function.
  /// @returns True if data is available.
  bool recvIpDataReadable();
  /// Get the next character of received IP data from the buffer.
  char getcRecvIpData();
  /// Clear all received data from the buffer.
  void clearRecvData();
  
private:
  static ESP8266 * defaultInstance; ///< Default instance support for use with mbed Sockets.
  
  mbed::Serial AT_intf;
  mbed::DigitalOut resetPin;
  mutable mbed::DigitalOut powerDownPin; ///< @note Mark as mutable for use in powered().
  mbed::CircularBuffer<char, 1024> recvIpDataBuffer; ///< Received IP data buffer.
  mbed::Serial * debugMsg;
  volatile bool parseRecvReset; ///< Indicates when AT interface received data parsers should be reset.
  
  /// Send raw AT data to the ESP8266.
  /// @param cmdString Data to send.
  /// @param expectEcho True if the ESP8266 will echo sent data back.
  CmdResult send_AT_data(const std::string & cmdString, const bool expectEcho);
  
  /// Attempts to read an entire line terminated with \r\n from the AT interface.
  /// \r will be preserved in the final string and \n will be stripped.
  /// @param line Buffer to store received characters in.
  /// @returns True if an entire line was read.
  bool read_line(std::string & line);
  
  /// Callback for when data is received on the AT interface.
  void recv_AT_data_cb();
  /// Parse the next character received on the AT interface checking for valid IP data.
  void parseRecvIpData(const char received);
  /// Parse the next character receive on the AT interface for the connection closed message.
  void parseRecvConnClosedMsg(const char received);
  
  /// Print a message on the debugging interface if setup.
  /// @param message Null terminated string.
  void printDbgMsg(const char * message);
};

#endif