/*
 *
 * Developed by xabicrespog@gmail.com
 *
 */

#include "MX25L4006E.h"

MX25L4006E::MX25L4006E(SPI *spi) : 
    cs(P_MEMCS)
    {
        this->spi = spi;
    }

bool MX25L4006E::init()
{
    uint32_t RDID;
    uint8_t RES;
    CMD_RES(&RES);
    if (RES != FLASH_RES) {
        printf("RES 0x%hx\r\n", RES);
        return false;
    }
    
    Initial_Spi(); 
    if (CMD_RDID(&RDID) == (ReturnMsg)FlashOperationSuccess && RDID == FLASH_RDID)
    {
        return true;
    }
    else
    {
        return false;
    }
}
/*
 --Common functions
 */

/*
 * Function:       Wait_Flash_WarmUp
 * Arguments:      None.
 * Description:    Wait some time until flash read / write enable.
 * Return Message: None.
 */
void MX25L4006E::Wait_Flash_WarmUp()
{
    uint32_t time_cnt = FlashFullAccessTime;
    while( time_cnt > 0 )
    {
        time_cnt--;
    }
}

/*
 * Function:       Initial_Spi
 * Arguments:      None
 * Description:    Initial spi flash state and wait flash warm-up
 *                 (enable read/write).
 * Return Message: None
 */
void MX25L4006E::Initial_Spi()
{
    // Wait flash warm-up
    Wait_Flash_WarmUp();
}

/*
 * Function:       CS_Low, CS_High
 * Arguments:      None.
 * Description:    Chip select go low / high.
 * Return Message: None.
 */
void MX25L4006E::CS_Low()
{
    cs = 0;
}

void MX25L4006E::CS_High()
{
    //?WPn = 1;
    cs = 1;
}

/*
 * Function:       InsertDummyCycle
 * Arguments:      dummy_cycle, number of dummy clock cycle
 * Description:    Insert dummy cycle of SCLK
 * Return Message: None.
 */
void MX25L4006E::InsertDummyCycle( uint8_t dummy_cycle )
{
    for( uint8_t i=0; i < dummy_cycle/8; i++ )
    {
        get_byte();
    }
}

/*
 * Function:       send_byte
 * Arguments:      byte_value, data transfer to flash
 * Description:    Send one byte data to flash
 * Return Message: 8 bit response from SPI master.
 */
uint8_t MX25L4006E::send_byte(uint8_t byte_value)
{
    return spi->write(byte_value);
}

/*
 * Function:       get_byte
 * Description:    Get one byte data to flash
 * Return Message: 8 bit data
 */

uint8_t MX25L4006E::get_byte()
{
    return spi->write(0);
}

/*
 * Function:       WaitFlashReady
 * Arguments:      ExpectTime, expected time-out value of flash operations.
 *                 No use at non-synchronous IO mode.
 * Description:    Synchronous IO:
 *                 If flash is ready return true.
 *                 If flash is time-out return false.
 *                 Non-synchronous IO:
 *                 Always return true
 * Return Message: true, false
 */
bool MX25L4006E::WaitFlashReady( uint32_t ExpectTime )
{
#ifndef NON_SYNCHRONOUS_IO
    uint32_t temp = 0;
    while( IsFlashBusy() )
    {
        if( temp > ExpectTime )
        {
            return false;
        }
        temp = temp + 1;
    }
       return true;
#else
    return true;
#endif
}

/*
 * Function:       IsFlashBusy
 * Arguments:      None.
 * Description:    Check status register WIP bit.
 *                 If  WIP bit = 1: return true ( Busy )
 *                             = 0: return false ( Ready ).
 * Return Message: true, false
 */
bool MX25L4006E::IsFlashBusy( void )
{
    uint8_t  gDataBuffer;

    CMD_RDSR( &gDataBuffer );
    if( (gDataBuffer & FLASH_WIP_MASK)  == FLASH_WIP_MASK )
        return true;
    else
        return false;
}

/*
 * Function:       IsFlashQIO
 * Arguments:      None.
 * Description:    If flash QE bit = 1: return true
 *                                 = 0: return false.
 * Return Message: true, false
 */
bool MX25L4006E::IsFlashQIO( void )
{
#ifdef FLASH_NO_QE_BIT
    return true;
#else
    uint8_t  gDataBuffer;
    CMD_RDSR( &gDataBuffer );
    if( (gDataBuffer & FLASH_QE_MASK) == FLASH_QE_MASK )
        return true;
    else
        return false;
#endif
}
/*
 * Function:       IsFlash4Byte
 * Arguments:      None
 * Description:    Check flash address is 3-byte or 4-byte.
 *                 If flash 4BYTE bit = 1: return true
 *                                    = 0: return false.
 * Return Message: true, false
 */
bool MX25L4006E::IsFlash4Byte( void )
{
#ifdef FLASH_CMD_RDSCUR
    #ifdef FLASH_4BYTE_ONLY
        return true;
    #elif FLASH_3BYTE_ONLY
        return false;
    #else
        uint8_t  gDataBuffer;
        CMD_RDSCUR( &gDataBuffer );
        if( (gDataBuffer & FLASH_4BYTE_MASK) == FLASH_4BYTE_MASK )
            return true;
        else
            return false;
    #endif
#else
    return false;
#endif
}
/*
 * Function:       SendFlashAddr
 * Arguments:      flash_address, 32 bit flash memory address
 *                 io_mode, I/O mode to transfer address
 *                 addr_4byte_mode,
 * Description:    Send flash address with 3-byte or 4-byte mode.
 * Return Message: None
 */
void MX25L4006E::SendFlashAddr( uint32_t flash_address, uint8_t io_mode, uint8_t addr_4byte_mode )
{
    /* Check flash is 3-byte or 4-byte mode.
       4-byte mode: Send 4-byte address (A31-A0)
       3-byte mode: Send 3-byte address (A23-A0) */
    if( addr_4byte_mode == true ){
        send_byte((flash_address >> 24)); // A31-A24
    }
    /* A23-A0 */
    send_byte((flash_address >> 16));
    send_byte((flash_address >> 8));
    send_byte((flash_address));
}

/*
 * Function:       GetDummyCycle
 * Arguments:      default_cycle, default dummy cycle
 *                 fsptr, pointer of flash status structure
 * Description:    Get dummy cycle for different condition
 *                 default_cycle: Byte3 | Byte2 | Byte1 | Byte0
 *                      DC 1 bit:   x       1       x       0
 *                      DC 2 bit:   11      10      01      00 
 *                 Note: the variable dummy cycle only support
                         in some product. 
 * Return Message: Dummy cycle value
 */
uint8_t MX25L4006E::GetDummyCycle( uint32_t default_cycle )
{
#ifdef FLASH_CMD_RDCR
    uint8_t gDataBuffer;
    uint8_t dummy_cycle = default_cycle;
    CMD_RDCR( &gDataBuffer );
    #ifdef SUPPORT_CR_DC
        // product support 1-bit dummy cycle configuration
        if( (gDataBuffer & FLASH_DC_MASK) == FLASH_DC_MASK )
            dummy_cycle = default_cycle >> 16;
        else
            dummy_cycle = default_cycle;
    #elif SUPPORT_CR_DC_2bit
        // product support 2-bit dummy cycle configuration
        switch( gDataBuffer & FLASH_DC_2BIT_MASK ){
            case 0x00:
                dummy_cycle = default_cycle;
            break;
            case 0x40:
                dummy_cycle = default_cycle >> 8;
            break;
            case 0x80:
                dummy_cycle = default_cycle >> 16;
            break;
            case 0xC0:
                dummy_cycle = default_cycle >> 24;
            break;
        }
    #else
         // configuration register not support dummy configuration
         dummy_cycle = default_cycle;
    #endif
    return dummy_cycle;
#else
    // default case: return default dummy cycle
    return default_cycle; 
#endif
}



/*
 * ID Command
 */

/*
 * Function:       CMD_RDID
 * Arguments:      Identification, 32 bit buffer to store id
 * Description:    The RDID instruction is to read the manufacturer ID
 *                 of 1-byte and followed by Device ID of 2-byte.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_RDID( uint32_t *RDID )
{
    uint32_t rx_data;

    // Chip select go low to start a flash command
    CS_Low();

    // Send command
    send_byte(FLASH_CMD_RDID);

    // Get answer (should be 0xC22013)
    rx_data = get_byte();
    rx_data = rx_data<<8;
    rx_data += get_byte();
    rx_data = rx_data<<8;
    rx_data += get_byte();    

    *RDID = rx_data;
    // Chip select go high to end a command
    CS_High();
   
    return FlashOperationSuccess;
}

/*
 * Function:       CMD_RES
 * Arguments:      ElectricIdentification, 8 bit buffer to store electric id
 * Description:    The RES instruction is to read the Device
 *                 electric identification of 1-byte.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_RES( uint8_t *ElectricIdentification )
{

    // Chip select go low to start a flash command
    CS_Low();

    // Send flash command and insert dummy cycle
    send_byte(FLASH_CMD_RES);
    
    InsertDummyCycle( 24 );

    // Get electric identification
    *ElectricIdentification = get_byte();

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}

/*
 * Function:       CMD_REMS
 * Arguments:      REMS_Identification, 16 bit buffer to store id
 *                 fsptr, pointer of flash status structure
 * Description:    The REMS instruction is to read the Device
 *                 manufacturer ID and electric ID of 1-byte.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_REMS( uint16_t *REMS_Identification, FlashStatus *fsptr )
{
    uint8_t  gDataBuffer[2];

    // Chip select go low to start a flash command
    CS_Low();

    // Send flash command and insert dummy cycle ( if need )
    // ArrangeOpt = 0x00 will output the manufacturer's ID first
    //            = 0x01 will output electric ID first
    send_byte(FLASH_CMD_REMS);
    InsertDummyCycle( 16 );
    send_byte(fsptr->ArrangeOpt);

    // Get ID
    gDataBuffer[0] = get_byte();
    gDataBuffer[1] = get_byte();

    // Store identification informaion
    *REMS_Identification = (gDataBuffer[0] << 8) | gDataBuffer[1];

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}


/*
 * Register  Command
 */

/*
 * Function:       CMD_RDSR
 * Arguments:      StatusReg, 8 bit buffer to store status register value
 * Description:    The RDSR instruction is for reading Status Register Bits.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_RDSR( uint8_t *StatusReg )
{
    uint8_t  gDataBuffer;

    // Chip select go low to start a flash command
    CS_Low();

    // Send command
    send_byte(FLASH_CMD_RDSR);
    gDataBuffer = get_byte();

    // Chip select go high to end a flash command
    CS_High();

    *StatusReg = gDataBuffer;

    return FlashOperationSuccess;
}

/*
 * Function:       CMD_WRSR
 * Arguments:      UpdateValue, 8/16 bit status register value to updata
 * Description:    The WRSR instruction is for changing the values of
 *                 Status Register Bits (and configuration register)
 * Return Message: FlashIsBusy, FlashTimeOut, FlashOperationSuccess
 */
#ifdef SUPPORT_WRSR_CR
MX25L4006E::ReturnMsg MX25L4006E::CMD_WRSR( uint16_t UpdateValue )
#else
MX25L4006E::ReturnMsg MX25L4006E::CMD_WRSR( uint8_t UpdateValue )
#endif
{
    // Check flash is busy or not
    if( IsFlashBusy() )    return FlashIsBusy;

    // Setting Write Enable Latch bit
    CMD_WREN();

    // Chip select go low to start a flash command
    CS_Low();

    // Send command and update value
    send_byte(FLASH_CMD_WRSR);
    send_byte(UpdateValue);
#ifdef SUPPORT_WRSR_CR
    send_byte(UpdateValue >> 8);    // write configuration register
#endif

    // Chip select go high to end a flash command
    CS_High();


    if( WaitFlashReady( WriteStatusRegCycleTime ) )
        return FlashOperationSuccess;
    else
        return FlashTimeOut;

}


/*
 * Read Command
 */

/*
 * Function:       CMD_READ
 * Arguments:      flash_address, 32 bit flash memory address
 *                 target_address, buffer address to store returned data
 *                 byte_length, length of returned data in byte unit
 * Description:    The READ instruction is for reading data out.
 * Return Message: FlashAddressInvalid, FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_READ( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length )
{
    uint32_t index;
    uint8_t  addr_4byte_mode;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;  // 4-byte mode
    else
        addr_4byte_mode = false; // 3-byte mode

    // Chip select go low to start a flash command
    CS_Low();

    // Write READ command and address
    send_byte(FLASH_CMD_READ);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );

    // Set a loop to read data into buffer
    for( index=0; index < byte_length; index++ )
    {
        // Read data one byte at a time
        *(target_address + index) = get_byte();
    }

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}


/*
 * Function:       CMD_DREAD
 * Arguments:      flash_address, 32 bit flash memory address
 *                 target_address, buffer address to store returned data
 *                 byte_length, length of returned data in byte unit
 * Description:    The DREAD instruction enable double throughput of Serial
 *                 Flash in read mode
 * Return Message: FlashAddressInvalid, FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_DREAD( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length )
{
    uint32_t index;
    uint8_t  addr_4byte_mode;
    uint8_t  dc;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;    // 4-byte mode
    else
        addr_4byte_mode = false;   // 3-byte mode

    // get dummy cycle number
    dc = GetDummyCycle( DUMMY_CONF_DREAD );

    // Chip select go low to start a flash command
    CS_Low();

    // Write 2-I/O Read command and address
    send_byte(FLASH_CMD_DREAD);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );
    InsertDummyCycle( dc );                    // Wait 8 dummy cycle

    // Set a loop to read data into data buffer
    for( index=0; index < byte_length; index++ )
    {
        *(target_address + index) = get_byte();
    }

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}


/*
 * Function:       CMD_FASTREAD
 * Arguments:      flash_address, 32 bit flash memory address
 *                 target_address, buffer address to store returned data
 *                 byte_length, length of returned data in byte unit
 * Description:    The FASTREAD instruction is for quickly reading data out.
 * Return Message: FlashAddressInvalid, FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_FASTREAD( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length )
{
    uint32_t index;
    uint8_t  addr_4byte_mode;
    uint8_t  dc;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;  // 4-byte mode
    else
        addr_4byte_mode = false; // 3-byte mode

    dc = GetDummyCycle( DUMMY_CONF_FASTREAD ); 

    // Chip select go low to start a flash command
    CS_Low();

    // Write Fast Read command, address and dummy cycle
    send_byte(FLASH_CMD_FASTREAD);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );
    InsertDummyCycle ( dc );          // Wait dummy cycle

    // Set a loop to read data into data buffer
    for( index=0; index < byte_length; index++ )
    {
        *(target_address + index) = get_byte();
    }

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}


/*
 * Function:       CMD_RDSFDP
 * Arguments:      flash_address, 32 bit flash memory address
 *                 target_address, buffer address to store returned data
 *                 byte_length, length of returned data in byte unit
 * Description:    RDSFDP can retrieve the operating characteristics, structure
 *                 and vendor-specified information such as identifying information,
 *                 memory size, operating voltages and timinginformation of device
 * Return Message: FlashAddressInvalid, FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_RDSFDP( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length )
{
    uint32_t index;
    uint8_t  addr_4byte_mode;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;  // 4-byte mode
    else
        addr_4byte_mode = false; // 3-byte mode

    // Chip select go low to start a flash command
    CS_Low();

    // Write Read SFDP command
    send_byte(FLASH_CMD_RDSFDP);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );
    InsertDummyCycle ( 8 );        // Insert dummy cycle

    // Set a loop to read data into data buffer
    for( index=0; index < byte_length; index++ )
    {
        *(target_address + index) = get_byte();
    }

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}
/*
 * Program Command
 */

/*
 * Function:       CMD_WREN
 * Arguments:      None.
 * Description:    The WREN instruction is for setting
 *                 Write Enable Latch (WEL) bit.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_WREN( void )
{
    // Chip select go low to start a flash command
    CS_Low();

    // Write Enable command = 0x06, Setting Write Enable Latch Bit
    send_byte(FLASH_CMD_WREN);

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}

/*
 * Function:       CMD_WRDI
 * Arguments:      None.
 * Description:    The WRDI instruction is to reset
 *                 Write Enable Latch (WEL) bit.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_WRDI( void )
{
    // Chip select go low to start a flash command
    CS_Low();

    // Write Disable command = 0x04, resets Write Enable Latch Bit
    send_byte(FLASH_CMD_WRDI);

    CS_High();

    return FlashOperationSuccess;
}


/*
 * Function:       CMD_PP
 * Arguments:      flash_address, 32 bit flash memory address
 *                 source_address, buffer address of source data to program
 *                 byte_length, byte length of data to programm
 * Description:    The PP instruction is for programming
 *                 the memory to be "0".
 *                 The device only accept the last 256 byte ( or 32 byte ) to program.
 *                 If the page address ( flash_address[7:0] ) reach 0xFF, it will
 *                 program next at 0x00 of the same page.
 *                 Some products have smaller page size ( 32 byte )
 * Return Message: FlashAddressInvalid, FlashIsBusy, FlashOperationSuccess,
 *                 FlashTimeOut
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_PP( uint32_t flash_address, uint8_t *source_address, uint32_t byte_length )
{
    uint32_t index;
    uint8_t  addr_4byte_mode;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check flash is busy or not
    if( IsFlashBusy() )    return FlashIsBusy;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;  // 4-byte mode
    else
        addr_4byte_mode = false; // 3-byte mode

    // Setting Write Enable Latch bit
    CMD_WREN();
    
    // Chip select go low to start a flash command
    CS_Low();
    
    // Write Page Program command
    send_byte(FLASH_CMD_PP);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );

    // Set a loop to down load whole page data into flash's buffer
    // Note: only last 256 byte ( or 32 byte ) will be programmed
    for( index=0; index < byte_length; index++ )
    {
        send_byte(*(source_address + index));
    }

    // Chip select go high to end a flash command
    CS_High();

    if( WaitFlashReady( PageProgramCycleTime ) )
        return FlashOperationSuccess;
    else
        return FlashTimeOut;
}


/*
 * Erase Command
 */

/*
 * Function:       CMD_SE
 * Arguments:      flash_address, 32 bit flash memory address
 * Description:    The SE instruction is for erasing the data
 *                 of the chosen sector (4KB) to be "1".
 * Return Message: FlashAddressInvalid, FlashIsBusy, FlashOperationSuccess,
 *                 FlashTimeOut
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_SE( uint32_t flash_address )
{
    uint8_t  addr_4byte_mode;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check flash is busy or not
    if( IsFlashBusy() )    return FlashIsBusy;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;  // 4-byte mode
    else
        addr_4byte_mode = false; // 3-byte mode

    // Setting Write Enable Latch bit
    CMD_WREN();

    // Chip select go low to start a flash command
    CS_Low();

    //Write Sector Erase command = 0x20;
    send_byte(FLASH_CMD_SE);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );

    // Chip select go high to end a flash command
    CS_High();

    if( WaitFlashReady( SectorEraseCycleTime ) )
        return FlashOperationSuccess;
    else
        return FlashTimeOut;
}


/*
 * Function:       CMD_BE
 * Arguments:      flash_address, 32 bit flash memory address
 * Description:    The BE instruction is for erasing the data
 *                 of the chosen sector (64KB) to be "1".
 * Return Message: FlashAddressInvalid, FlashIsBusy, FlashOperationSuccess,
 *                 FlashTimeOut
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_BE( uint32_t flash_address )
{
    uint8_t  addr_4byte_mode;

    // Check flash address
    if( flash_address > FlashSize ) return FlashAddressInvalid;

    // Check flash is busy or not
    if( IsFlashBusy() )    return FlashIsBusy;

    // Check 3-byte or 4-byte mode
    if( IsFlash4Byte() )
        addr_4byte_mode = true;  // 4-byte mode
    else
        addr_4byte_mode = false; // 3-byte mode

    // Setting Write Enable Latch bit
    CMD_WREN();

    // Chip select go low to start a flash command
    CS_Low();

    //Write Block Erase command = 0xD8;
    send_byte(FLASH_CMD_BE);
    SendFlashAddr( flash_address, SIO, addr_4byte_mode );

    // Chip select go high to end a flash command
    CS_High();

    if( WaitFlashReady( BlockEraseCycleTime ) )
        return FlashOperationSuccess;
    else
        return FlashTimeOut;
}

/*
 * Function:       CMD_CE
 * Arguments:      None.
 * Description:    The CE instruction is for erasing the data
 *                 of the whole chip to be "1".
 * Return Message: FlashIsBusy, FlashOperationSuccess, FlashTimeOut
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_CE( void )
{
    // Check flash is busy or not
    if( IsFlashBusy() )    return FlashIsBusy;

    // Setting Write Enable Latch bit
    CMD_WREN();

    // Chip select go low to start a flash command
    CS_Low();

    //Write Chip Erase command = 0x60;
    send_byte(FLASH_CMD_CE);

    // Chip select go high to end a flash command
    CS_High();

    if( WaitFlashReady( ChipEraseCycleTime ) )
        return FlashOperationSuccess;
    else
        return FlashTimeOut;
}


/*
 * Mode setting Command
 */

/*
 * Function:       CMD_DP
 * Arguments:      None.
 * Description:    The DP instruction is for setting the
 *                 device on the minimizing the power consumption.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_DP( void )
{
    // Chip select go low to start a flash command
    CS_Low();

    // Deep Power Down Mode command
    send_byte(FLASH_CMD_DP);

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}

/*
 * Function:       CMD_RDP
 * Arguments:      None.
 * Description:    The Release from RDP instruction is
 *                 putting the device in the Stand-by Power mode.
 * Return Message: FlashOperationSuccess
 */
MX25L4006E::ReturnMsg MX25L4006E::CMD_RDP( void )
{
    // Chip select go low to start a flash command
    CS_Low();

    // Deep Power Down Mode command
    send_byte(FLASH_CMD_RDP);

    // Chip select go high to end a flash command
    CS_High();

    return FlashOperationSuccess;
}


/*
 * Reset setting Command
 */

/*
 * Security Command
 */

/*
 * Suspend/Resume Command
 */
