/*
 *
 * Developed by xabicrespog@gmail.com
 *
 */

#ifndef    __MX25_H__
#define    __MX25_H__

#include "mbed.h"
#include "pins.h"

// beginning of MX25_DEF
/* Note:
   Synchronous IO     : MCU will polling WIP bit after
                        sending prgram/erase command
   Non-synchronous IO : MCU can do other operation after
                        sending prgram/erase command
   Default is synchronous IO
*/
//#define    NON_SYNCHRONOUS_IO

#define    FLASH_RDID        0xC22013
#define    FLASH_RES         0x12
#define    FLASH_REMS        0xC212

// variable
#define    BYTE_LEN          8
#define    IO_MASK           0x80
#define    HALF_WORD_MASK    0x0000ffff

/*
  Flash Related Parameter Define
*/
#define    Block_Offset       0x10000     // 64K Block size
#define    Block32K_Offset    0x8000      // 32K Block size
#define    Sector_Offset      0x1000      // 4K Sector size
#define    Page_Offset        0x0100      // 256 Byte Page size
#define    Page32_Offset      0x0020      // 32 Byte Page size (some products have smaller page size)
#define    Block_Num          (FlashSize / Block_Offset)

// Flash control register mask define
// status register
#define    FLASH_WIP_MASK         0x01
#define    FLASH_LDSO_MASK        0x02
#define    FLASH_QE_MASK          0x40
// security register
#define    FLASH_OTPLOCK_MASK     0x03
#define    FLASH_4BYTE_MASK       0x04
#define    FLASH_WPSEL_MASK       0x80
// configuration reigster
#define    FLASH_DC_MASK          0x80
#define    FLASH_DC_2BIT_MASK     0xC0
// other
#define    BLOCK_PROTECT_MASK     0xff
#define    BLOCK_LOCK_MASK        0x01

// dummy cycle configration
#define    DUMMY_CONF_2READ       0x0A080604
#define    DUMMY_CONF_4READ       0x0A080406
#define    DUMMY_CONF_FASTREAD    0x0A080608
#define    DUMMY_CONF_DREAD       0x0A080608
#define    DUMMY_CONF_QREAD       0x0A080608

/* Timer Parameter */
//#define  TIMEOUT    1
//#define  TIMENOTOUT 0
//#define  TPERIOD    240             // ns, The time period of timer count one
//#define  COUNT_ONE_MICROSECOND  16  // us, The loop count value within one micro-second


/*
  System Information Define
*/

#define    CLK_PERIOD               100   // unit: ns
#define    Min_Cycle_Per_Inst       1      // cycle count of one instruction
#define    One_Loop_Inst            4       // instruction count of one loop (estimate)

/*
  Flash ID, Timing Information Define
  (The following information could get from device specification)
*/

#define    FlashID          0xc22013
#define    ElectronicID     0x12
#define    RESID0           0xc212
#define    RESID1           0x12c2
#define    FlashSize        0x80000        // 512 KBytes
#define    CE_period        3906250        // tCE / ( CLK_PERIOD * Min_Cycle_Per_Inst *One_Loop_Inst)
#define    tW               100000000      // N*15ms, N is a multiple of 10K cycle
#define    tDP              10000          // 10us
#define    tPP              5000000        // 5ms
#define    tSE              60000000       // 60ms
#define    tBE              2000000000     // 2s
#define    tPUW             10000000       // 10ms
// Support I/O mode
#define    SIO              0
#define    DIO              1

// Flash information define
#define    WriteStatusRegCycleTime     tW / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#define    PageProgramCycleTime        tPP / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#define    SectorEraseCycleTime        tSE / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#define    BlockEraseCycleTime         tBE / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#define    ChipEraseCycleTime          CE_period
#define    FlashFullAccessTime         tPUW / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)

#ifdef tBP
#define    ByteProgramCycleTime        tBP / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#endif
#ifdef tWSR
#define    WriteSecuRegCycleTime       tWSR / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#endif
#ifdef tBE32
#define    BlockErase32KCycleTime      tBE32 / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#endif
#ifdef tWREAW
#define    WriteExtRegCycleTime        tWREAW / (CLK_PERIOD * Min_Cycle_Per_Inst * One_Loop_Inst)
#endif
// end of MX25_DEF.h

/*** MX25 series command hex code definition ***/
//ID comands
#define    FLASH_CMD_RDID        0x9F    //RDID (Read Identification)
#define    FLASH_CMD_RES         0xAB    //RES (Read Electronic ID)
#define    FLASH_CMD_REMS        0x90    //REMS (Read Electronic & Device ID)

//Register comands
#define    FLASH_CMD_WRSR        0x01    //WRSR (Write Status Register)
#define    FLASH_CMD_RDSR        0x05    //RDSR (Read Status Register)

//READ comands
#define    FLASH_CMD_READ        0x03    //READ (1 x I/O)
#define    FLASH_CMD_FASTREAD    0x0B    //FAST READ (Fast read data)
#define    FLASH_CMD_DREAD       0x3B    //DREAD (1In/2 Out fast read)
#define    FLASH_CMD_RDSFDP      0x5A    //RDSFDP (Read SFDP)

//Program comands
#define    FLASH_CMD_WREN        0x06    //WREN (Write Enable)
#define    FLASH_CMD_WRDI        0x04    //WRDI (Write Disable)
#define    FLASH_CMD_PP          0x02    //PP (page program)

//Erase comands
#define    FLASH_CMD_SE          0x20    //SE (Sector Erase)
#define    FLASH_CMD_BE          0xD8    //BE (Block Erase)
#define    FLASH_CMD_CE          0x60    //CE (Chip Erase) hex code: 60 or C7

//Mode setting comands
#define    FLASH_CMD_DP          0xB9    //DP (Deep Power Down)
#define    FLASH_CMD_RDP         0xAB    //RDP (Release form Deep Power Down)
#ifdef SBL_CMD_0x77
#else
#endif

//Reset comands

//Security comands
#ifdef LCR_CMD_0xDD_0xD5
#else
#endif

class MX25L4006E
{
public:
    MX25L4006E(SPI *spi);
    bool init();

    // Return Message
    typedef enum {
        FlashOperationSuccess,
        FlashWriteRegFailed,
        FlashTimeOut,
        FlashIsBusy,
        FlashQuadNotEnable,
        FlashAddressInvalid
    } ReturnMsg;

    // Flash status structure define
    struct sFlashStatus{
        /* Mode Register:
         * Bit  Description
         * -------------------------
         *  7   RYBY enable
         *  6   Reserved
         *  5   Reserved
         *  4   Reserved
         *  3   Reserved
         *  2   Reserved
         *  1   Parallel mode enable
         *  0   QPI mode enable
        */
        uint8_t    ModeReg;
        bool     ArrangeOpt;
    };

    typedef struct sFlashStatus FlashStatus;

    /* Basic functions */
    void CS_High();
    void CS_Low();
    void InsertDummyCycle( uint8_t dummy_cycle );

    /* Utility functions */
    void Wait_Flash_WarmUp();
    void Initial_Spi();
    bool WaitFlashReady( uint32_t ExpectTime );
    bool WaitRYBYReady( uint32_t ExpectTime );
    bool IsFlashBusy( void );
    bool IsFlashQIO( void );
    bool IsFlash4Byte( void );
    void SendFlashAddr( uint32_t flash_address, uint8_t io_mode, uint8_t addr_mode );
    uint8_t GetDummyCycle( uint32_t default_cycle );

    /* Flash commands */
    ReturnMsg CMD_RDID( uint32_t *RDID );
    ReturnMsg CMD_RES( uint8_t *ElectricIdentification );
    ReturnMsg CMD_REMS( uint16_t *REMS_Identification, FlashStatus *fsptr );

    ReturnMsg CMD_RDSR( uint8_t *StatusReg );
    #ifdef SUPPORT_WRSR_CR
       ReturnMsg CMD_WRSR( uint16_t UpdateValue );
    #else
       ReturnMsg CMD_WRSR( uint8_t UpdateValue );
    #endif

    ReturnMsg CMD_READ( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length  );
    ReturnMsg CMD_FASTREAD( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length );
    ReturnMsg CMD_DREAD( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length );
    ReturnMsg CMD_RDSFDP( uint32_t flash_address, uint8_t *target_address, uint32_t byte_length );

    ReturnMsg CMD_WREN( void );
    ReturnMsg CMD_WRDI( void );
    ReturnMsg CMD_PP( uint32_t flash_address, uint8_t *source_address, uint32_t byte_length );

    ReturnMsg CMD_SE( uint32_t flash_address );
    ReturnMsg CMD_BE( uint32_t flash_address );
    ReturnMsg CMD_CE( void );

    ReturnMsg CMD_DP( void );
    ReturnMsg CMD_RDP( void );

private:
    SPI *spi;

    DigitalOut cs;
    uint8_t send_byte(uint8_t byte_value);
    uint8_t get_byte();

};

void cmd_rdid_callback(void);

#endif    /* __MX25_H__ */
