/** testSensor
 * I2C and I2C sensor test program
 * For the initial release, 
 * following MSU sensors are supported
 * FXOS8700CQ 
 * HIH6130
 * MAG3110
 * MMA8451Q
 * MMA8452Q
 * MPL3115A2
 * MAX44000
 * MAX44005 (not tested)
 * MAX44008
 * 
 * For untested sensor(s) or I2C devices
 * open / close / read / write 
 * commands are avilable
 * please refer to the help message and source code 
 * for details.
 */
#include "mbed.h"
#include <string.h>
#include <ctype.h>
#include "vt100.h"
#include "MSS.h"
#include "MSU.h"
#include "dumb_i2c.h"
#define TEST_LOOP 10

DUMB_I2C *i2c = 0 ;
vt100 *tty = 0 ;
int test_loop = TEST_LOOP ;
int interval = 100 ; /* ms wait interval */

void doSWReset(void)
{
    printf("Software Reset\n") ;
    SCB->AIRCR = 0x05FA0004 ;
}

void doClose(void)
{
    if (i2c != 0) {
        printf("Closing I2C at 0x%02X ... ", i2c->address()) ;
        delete i2c ;
        i2c = 0 ;
        printf("Done\n") ;
    }
}

void doOpen(void)
{
    uint8_t address ;
    scanf("%X", &address) ;
    if (i2c != 0) {
        doClose() ;
    }
    printf("Opening I2C at 0x%02X ... ", address) ;
    i2c = new DUMB_I2C(PIN_SDA, PIN_SCL, address) ;
    printf("Done\n") ;
}

void doRead(void)
{
    int addr ;
    int len ;
    uint8_t *data ;
    int result ;
    if (i2c == 0) {
        printf("I2C is not opened\n") ;
        return ;
    }
    scanf("%X %X", &addr, &len) ;
    if (len > 0) {
        data = new uint8_t[len] ;
    }
 //   i2c->read(addr, data, len) ;
    printf("0x%02X : ", (unsigned int)addr) ;
    for (int i = 0 ; i < len ; i++ ) {
        result = i2c->read(addr+i, &data[i], 1) ;
        if (result == 0) {
            printf("%02X ", data[i]) ;
            if (((i+1) < len)&&(((i+1)%0x10) == 0)) {
                printf("\n") ;
                printf("0x%02X : ", (unsigned int)(addr + i + 1)) ;
            }
        } else {
            printf("failed to read 0x%02X\n", addr+i) ;
        }  
    }
    printf("\n") ;
}

void doWrite(void)
{
    int addr ;
    uint8_t len ;
    uint8_t data ;
    int ack ;
    if (i2c == 0) {
        printf("I2C is not opened\n") ;
        return ;
    }
    scanf("%X %X", &addr, &data) ;
    ack = i2c->write(addr, &data, 1) ;
    if (ack != 0) {
        printf("NAK at writing data[0x%02X] to address[0x%02X]\n",
            data, addr ) ;
    }
}

void doStatus(void)
{
    if (i2c == 0) {
        printf("i2c is not opened\n") ;
    } else {
        printf("i2c device at 0x%02X is opened at %d Hz\n", 
            i2c->address(), i2c->frequency()) ;
    }
    printf("test loop number = %d\n", test_loop) ;
    printf("loop interval = %d ms\n", interval) ;
}

void doHelp(void)
{
    printf("Simple I2C test program %s for %s\n", __DATE__, BOARD_NAME) ;
    printf("\n             === usage ===\n") ;
    printf("open <7bit addr>    : open i2c device at <7bit addr>\n") ;
    printf("close               : close currently opened i2c\n") ;
    printf("read <addr> <len>   : read <len> data from <addr>\n") ;
    printf("write <addr> <data> : write <data> to register <addr>\n") ;
    printf("frequency <freq>    : change frequency to <freq> Hz\n") ;
    printf("bus                 : bus scan for existing I2C addresses\n") ;
    printf("test <sensor or address>  : test a sensor\n") ;
    printf("loop <number>       : specify loop count for test\n") ;
    printf("interval <numver>   : ms interval for each loop\n") ;
    printf("status              : print current status\n") ;
    printf("demo                : mulitple sensor demo\n") ;
    printf("help                : print this help\n") ;
    printf("zzz                 : cause software reset\n") ;
    printf("\nPlease set local-echo to see what you are typing.\n") ;
    printf("\n") ; 
}

void doFreq(void)
{
    int freq = 0 ;
    scanf("%d", &freq) ;
    if (i2c != 0) {
        i2c->frequency(freq) ;
    }
}


void print_sensor_name(int address)
{
    int i ;
    for(i = 0; i2c_sensor[i].address != 0 ; i++) {
        if (i2c_sensor[i].address == address) {
            printf("%s ", i2c_sensor[i].name) ;
        }
    }
}

void doBusScan(void)
{
    int address ;
    uint8_t data[10] ;
    int num_data = 1 ;
    int result ;
    if (i2c != 0) {
        printf("Closing I2C at 0x%02X ... ", i2c->address()) ;
        delete i2c ;
        i2c = 0 ;
        printf("Done\n") ;
    }

    for (address = 1; address < 127 ; address++) {
        i2c = new DUMB_I2C(PIN_SDA, PIN_SCL, address) ;
        num_data = 2 ;
        result = i2c->read(0, data, num_data) ;
/*
        data[0] = address << 1 ;
        i2c->start() ;
        result = i2c->write(0, data, 1) ;
        i2c->stop() ;
*/
        if (result == 0) {
            printf("%02X : ", address) ;
            print_sensor_name(address) ;
            printf("\n") ;
        }
        delete i2c ;
        i2c = 0 ; 
    }
    printf("\n") ;
}

void setTestLoop(void)
{
    int num ;
    scanf("%d", &num) ;
    if (num < 0) { num = 1 ; }
    test_loop = num ;
    printf("test loop count set to %d\n", test_loop) ;
}

void setTestInterval(void)
{
    int num ;
    scanf("%d", &num) ;
    if (num < 0) num = 100 ;
    interval = num ;
    printf("wait %d ms for each loop\n", interval) ;
}

void str2upper(char *str)
{
    while(str && *str) {
        if (('a' <= *str) && (*str <= 'z')) {
            *str -= 'a' - 'A' ;
        }
        str++ ;
    }
}

void doTestSensor(void) 
{
    int i ;
    int address = 0 ;
    char name[32] ;
    scanf("%s", name) ;
    str2upper(name) ;
    if (('0' <= *name)&&(*name <= '9')) { /* assume it's the address */
        sscanf(name, "%X", &address) ;
        for (i = 0 ; i < *(i2c_sensor[i].name) != 0 ; i++ ) {
            if (i2c_sensor[i].address == address) {
                strcpy(name, i2c_sensor[i].name) ;
                break ;
            }
        }
    } else {
        for (i = 0 ; i2c_sensor[i].address != 0 ; i++) {
            if (strcmp(name, i2c_sensor[i].name) == 0) { /* found */
                break ;
            }
        }
    }
    if (i2c_sensor[i].name != 0) { /* name found */
        i2c_sensor[i].test_func() ;
    }

}
    
void doCommand(char *str)
{
    switch(*str) {
    case 'o': case 'O': /* open */
        doOpen()    ; break ;
    case 'c': case 'C': /* close */
        doClose()   ; break ;
    case 'd': case 'D': /* demo */
        doDemo()    ; break ;
    case 'r': case 'R': /* read */
        doRead()    ; break ;
    case 'w': case 'W': /* write */
        doWrite()   ; break ;
    case 's': case 'S': /* status */
        doStatus()  ; break ;
    case 'f': case 'F': /* Frequency */
        doFreq()    ; break ;
    case 't': case 'T': /* test sensor */
        doTestSensor() ; break ;
    case 'l': case 'L': /* set test_loop */
        setTestLoop() ; break ;
    case 'i': case 'I': /* set interval */
        setTestInterval() ; break ;
    case 'b': case 'B': /* Bus Scan */
        doBusScan() ; break ;
    case 'z': case 'Z': /* Software Reset */
        doSWReset() ; break ;
    default:
        doHelp() ; break ;
    }
}

int main() {
    char cmd[32] ;
    uint32_t baud =  115200 ;
#if defined (TARGET_MAX32600MBED)
baud = 57600 ;
#endif
    tty = new vt100(baud) ;
    tty->cls() ;
    doHelp() ;
    while(1) {
        printf("> ") ;
        scanf("%s", cmd) ;
        doCommand(cmd) ;
    }
}
