#include "mbed.h"
#include "vt100.h"
#include "MSS.h"
#include "DS1683.h"

#define MSU_DS1683_ADDRESS 0x6B

vt100 *tty = 0 ;
DS1683 *ds1683 = 0 ;
int ds_alarm_flag = 0 ;
DigitalOut ledD1(PIN_SCK, 1) ; 
DigitalOut ledD2(PIN_MISO, 1) ;
DigitalOut ledD3(PIN_MOSI, 1) ;
DigitalOut ledD4(PIN_CS0, 1) ;

void doHelp(void)
{
    printf("DS1683 test program %s\n", __DATE__) ;
    printf("event <count>        : event count test\n") ;
    printf("time <sec>           : elapsed time count test\n") ;
    printf("clear                : clear event and etc registers\n") ;
    printf("mem                  : dump memory\n") ;
    printf("write <addr> <value> : write byte data to address\n") ;
    printf("help                 : print this\n") ;
    printf("\n") ;
}

void ds_alarm(void)
{
    ds_alarm_flag = 1 ;
}

void dump_mem(void)
{
    int i ;
    uint8_t data[0x30] ;
    if (ds1683) {
        ds1683->read(0, data, 0x30) ;
#if 0
        printf("   : ") ;
        for (i = 0 ; i < 0x10 ; i++ ) {
            printf("%02X ", i) ;
        }
        printf("\n") ;
        for (i = 0 ; i < 0x30 ; i++ ) {
            if ((i % 0x10)==0) {
                printf("%02X : ", i) ;
            }
            printf("%02X ", data[i]) ;
            if (((i+1) % 0x10)==0) {
                printf("\n") ;
            }
        }
        printf("\n") ;
#else
        printf("EVENT PIN: ") ;
        if (data[1] & 0x04) {
            printf("ON") ;
        } else {
            printf("OFF") ;
        }
        printf("\n") ;
        printf("EVENT Alarm: ") ;
        if (data[1] & 0x02) {
            printf("[ON]  ") ;
        } else {
            printf("[OFF] ") ;
        }
        printf("Event Counter: %02X%02X / ", data[9], data[8]) ;
        printf("%02X%02X\n", data[0x11], data[0x10]) ; 
        printf("ETC   Alarm: ") ;
        if (data[1] & 0x01) {
            printf("[ON]  ") ;
        } else {
            printf("[OFF] ") ;
        }
        printf("ETC   Counter: %02X%02X%02X%02X / ", 
            data[0x0D], data[0x0C], data[0x0B], data[0x0A]) ;
        printf("%02X%02X%02X%02X\n",
            data[0x15], data[0x14], data[0x13], data[0x12]) ;
        printf("\n") ;
#endif
    }
}

void init_hardware(void)
{
#if defined (TARGET_MAX32600MBED)
    tty = new vt100(57600) ;
    DigitalIn tmpIn(P2_5, PullUp) ;
    DigitalOut tmpOut(P1_2) ;
        ds1683 = new DS1683(
        PIN_SDA,  /* I2C SDA */
        PIN_SCL,  /* I2C SCL */
        P1_2,     /* event GPIO output */ /* MAX32600 only */
        P2_5, /* alarm interrupt */
        MSU_DS1683_ADDRESS ) ; /* 0x6B as 7bit address */
#else
    tty = new vt100() ;
    DigitalIn tmpIn(PIN_INT0, PullUp) ;
    DigitalOut tmpOut(PIN_AN0) ;
        ds1683 = new DS1683(
        PIN_SDA,  /* I2C SDA */
        PIN_SCL,  /* I2C SCL */
        PIN_AN0,  /* event GPIO output */
        PIN_INT0, /* alarm interrupt */
        MSU_DS1683_ADDRESS ) ; /* 0x6B as 7bit address */
#endif

    ds1683->enterPW(0xFFFFFFFF) ;
    tty->cls() ;
}

/* test Elapsed Time Counter */
void doETC(int sec)
{
    int loop = 0 ;
    int time_out ;
    ds_alarm_flag = 0 ;
    ds1683->event = 0 ; /* set event pin inactive */
    ds1683->clearETC() ; /* clear elapsed time counter */
    ds1683->setETCAlarm(sec * 4) ; /* etc alarm limit sec x 250msdiv(4) */
    time_out = sec + 5 ;
    ds1683->enableETCAlarm() ;
    
    printf("ETC (Elapsed Time Count) Test for %d sec\n", sec) ;
    ds1683->event = 1 ; /* activate event pin */
    while(ds_alarm_flag == 0) {
        ledD3 = 0 ;
        wait(0.5) ;
        ledD3 = 1 ;
        wait(0.5) ;
        printf("%d sec\n", loop++) ;  fflush(stdout) ;

        if (ds1683->getStatus() & 0x01) { /* ETC_AF_BIT */
            printf("ETC alarm detected\n") ;
            ledD1 = 0 ; /* LED1 ON */
            break ;
        } else if (loop > time_out) {
            printf("ETC Test Time Out\n") ;
            break ;
        }
    }
    ds1683->event = 0 ;
    ds_alarm_flag = 0 ;
}
    
/* test Event Counter */
void doEvent(int num)
{
    int loop = 0 ;
    int time_out ;
    ds_alarm_flag = 0 ;
    ds1683->disableEventAlarm() ; 
    ds1683->clearEvent() ;
    ds1683->clearAlarm() ;
    ds1683->setEventAlarm(num) ;
    time_out = num + 5 ;
    printf("Event count test for %d events\n", num) ;
    ds1683->enableEventAlarm() ;
    ds1683->event.write(1) ;
    while(ds_alarm_flag == 0) {
        ledD4 = 0 ; /* LED ON */
        ds1683->event.write(0) ;
        printf("%d ", loop++) ; fflush(stdout) ;
        wait(0.4) ;
        ds1683->event.write(1) ;
        ledD4 = 1 ; /* LED OFF */
        if (ds1683->getStatus() & 0x02) { /* EVENT_AF_BIT */
            printf("\nEvent count alarm detected\n") ;
            ledD2 = 0 ; /* LED2 ON */
            break ;
        } else if (loop > time_out) {
            printf("\nEvent count test time out\n") ;
            break ;
        }
        wait(0.1) ;
    }
    ds1683->event = 0 ;
    ds_alarm_flag = 0 ;
}

void doCommand(char *cmd)
{
    int count = 0 ;
    int addr ;
    int data ;
    
    switch(*cmd) {
    case 'e': case 'E': /* event */
        scanf("%d", &count) ;
        doEvent(count) ;
        break ;
    case 't': case 'T': /* ETC */
        scanf("%d", &count) ;
        doETC(count) ;
        break ;
    case 'm': case 'M': /* dump Memory */
        dump_mem() ;
        break ;
    case 'c': case 'C': /* clear registers */
        ds1683->clearRegs() ;
        break ;
    case 'w': case 'W': /* write byte */
        scanf("%x %x",&addr, &data) ;
        ds1683->writeReg8(addr, data) ;
        break ;
    case 'p': case 'P': /* event pin */
        scanf("%d", &data) ;
        if (data) {
            ds1683->event = 1 ;
            ledD4 = 0 ;
        } else { 
            ds1683->event = 0 ;
            ledD4 = 1 ;
        }
        break ;
    default:
        doHelp() ;
        break ;
    }
}

void doLEDs(void)
{
    uint8_t status ;
    status = ds1683->getStatus() ;
    if (status & 0x01) {
        ledD1 = 0 ;
    } else {
        ledD1 = 1 ;
    }
    if (status & 0x02) {
        ledD2 = 0 ;
    } else {
        ledD2 = 1 ;
    }
}

void prompt(void)
{
    printf("> ") ;
    fflush(stdout) ;
    wait(0.01) ;
    doLEDs() ;
}

int main() {
    char cmd[32] ;
    
    init_hardware() ;
    
    doHelp() ;
    prompt() ;
    while(1) {
        scanf("%s", cmd) ;
        doCommand(cmd) ;
        prompt() ;
    }
}
