#include "mbed.h"
#include "VCNL4100.h"

/* I2C protocol
 *
 * I2C bus timing for Sendign Word Command Format 
 * <s> <8bit: Slave Address>W <sa> <8bit: command code> <sa>
 * <8bit: Data Byte Low> <sa> <8bit: Data Byte High> <sa> <p>
 *
 * I2C bus timing for Receiving Word Command Format
 * <s> <8bit: Slave Address>W <sa> <8bit: Command Code> <sa>
 * <s> <8bit: Slave Address>R <sa> <8bit: Data Byte Low> <ma>
 * <8bit: Data Byte High> <ma> <p>
 * 
 * <s> Start by Master
 * <p> Stop by Master
 * <ma> ACK by Master
 * <sa> ACK by Slave (VCNL4100)
 */
 
/* Internal Registers */
#define REG_ALS_CONF   0x00
#define REG_ALS_THDH_L 0x01
#define REG_ALS_THDL_L 0x02
#define REG_PS_CONF1   0x03
#define REG_PS_CONF3   0x04
#define REG_PS_THDL    0x06
#define REG_PS_DATA    0x08
#define REG_ASL_DATA   0x09
#define REG_INT_FLAG   0x0B

VCNL4100::VCNL4100(PinName sda, PinName scl, int addr) :
    m_i2c(sda, scl), m_addr(addr<<1) 
{
    // activate the peripheral
}


VCNL4100::~VCNL4100()
{
}

void VCNL4100::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void VCNL4100::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}

uint8_t  VCNL4100::getAlsConf(void) 
{
    uint8_t data[2] ;
    readRegs(REG_ALS_CONF, data, 2) ;
    return(data[0]) ;
}

void     VCNL4100::setAlsConf(uint8_t conf) 
{
    uint8_t data[3] ;
    data[0] = REG_ALS_CONF ;
    data[1] = conf ;
    data[2] = 0x00 ;
    writeRegs(data, 3) ;
}

uint16_t VCNL4100::getAlsThdh(void) 
{
    uint16_t thdh ;
    uint8_t data[2] ;
    readRegs(REG_ALS_THDH_L, data, 2) ;
    thdh = (data[1] << 8) | data[0] ;
    return( thdh ) ;
}

void     VCNL4100::setAlsThdh(uint16_t thdh) 
{
    uint8_t data[3] ;
    data[0] = REG_ALS_THDH_L ;
    data[1] = thdh & 0xFF ;
    data[2] = (thdh >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

uint16_t VCNL4100::getAlsThdl(void) 
{
    uint16_t thdl ;
    uint8_t data[2] ;
    readRegs(REG_ALS_THDL_L, data, 2) ;
    thdl = (data[1] << 8) | data[0] ;
    return( thdl ) ;
}

void     VCNL4100::setAlsThdl(uint16_t thdl) 
{
    uint8_t data[3] ;
    data[0] = REG_ALS_THDL_L ;
    data[1] = thdl & 0xFF ;
    data[2] = (thdl >> 8) & 0xFF ;
    writeRegs(data, 3) ;    
}

uint16_t VCNL4100::getPsConf12(void) 
{
    uint16_t psconf12 ;
    uint8_t data[2] ;
    readRegs(REG_PS_CONF1, data, 2) ;
    psconf12 = (data[1] << 8) | data[0] ;
    return( psconf12 ) ;
}

void     VCNL4100::setPsConf12(uint16_t conf12) 
{
    uint8_t data[3] ;
    data[0] = REG_PS_CONF1 ;
    data[1] = conf12 & 0xFF ;
    data[2] = (conf12 >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

uint8_t  VCNL4100::getPsConf1(void) 
{
    uint16_t tmp ;
    uint8_t psconf1 ;
    tmp = getPsConf12() ;
    psconf1 = tmp & 0xFF ;
    return( psconf1 ) ;
}

void     VCNL4100::setPsConf1(uint8_t conf1) 
{
    uint16_t tmp ;
    uint8_t data[3] ;
    data[0] = REG_PS_CONF1 ;
    tmp = getPsConf12() ;
    data[1] = conf1 ;
    data[2] = (tmp >> 8) & 0xFF ; // conf2
    writeRegs(data, 3) ;
}

uint8_t  VCNL4100::getPsConf2(void) 
{
    uint16_t tmp ;
    uint8_t psconf2 ;
    tmp = getPsConf12() ;
    psconf2 = (tmp >> 8) & 0xFF ;
    return( psconf2 ) ;
}

void     VCNL4100::setPsConf2(uint8_t conf2) 
{
    uint16_t tmp ;
    uint8_t data[3] ;
    data[0] = REG_PS_CONF1 ;
    tmp = getPsConf12() ;
    data[1] = tmp & 0xFF ;
    data[2] = conf2 ;
    writeRegs(data, 3) ;
}

uint16_t VCNL4100::getPsConf3Spo(void) 
{
    uint16_t conf3spo ;
    uint8_t data[2] ; /* data[0] = PS_CONF3 ; data[1] = SPO ; */
    readRegs(REG_PS_CONF3, data, 2) ;
    conf3spo = (data[1] << 8) | data[0] ;
    return( conf3spo ) ;
}

void     VCNL4100::setPsConf3Spo(uint16_t conf3spo) 
{
    uint8_t data[3] ;
    data[0] = REG_PS_CONF3 ;
    data[1] = conf3spo & 0xFF ; /* PS_CONF3 */
    data[2] = (conf3spo >> 8) & 0xFF ; /* PS_SPO */
    writeRegs(data, 3) ;
}

uint8_t  VCNL4100::getPsConf3(void) 
{
    uint16_t conf3spo ;
    uint8_t conf3 ;
    conf3spo = getPsConf3Spo() ;
    conf3 = conf3spo & 0xFF ;
    return( conf3 ) ;
}

void     VCNL4100::setPsConf3(uint8_t conf3)
{
    uint16_t conf3spo ;
    uint8_t data[3] ;
    conf3spo = getPsConf3Spo() ;
    data[0] = REG_PS_CONF3 ;
    data[1] = conf3 ;
    data[2] = (conf3spo >> 8) & 0xFF ; /* spo */
    writeRegs(data, 3) ;
}

uint8_t  VCNL4100::getSpo(void) 
{
    uint16_t conf3spo ;
    uint8_t spo ;
    conf3spo = getPsConf3Spo() ;
    spo = (conf3spo >> 8) & 0xFF ;
    return( spo ) ;
}

void     VCNL4100::setSpo(uint8_t spo) 
{
    uint16_t conf3spo ;
    uint8_t data[3] ;
    conf3spo = getPsConf3Spo() ;
    data[0] = REG_PS_CONF3 ;
    data[1] = conf3spo & 0xFF ; /* PS_CONF3 */
    data[2] = spo ; /* PS_SPO */
    writeRegs(data, 3) ;
}

uint16_t VCNL4100::getPsThd(void) 
{
    uint16_t psthd ;
    uint8_t data[2] ;
    readRegs(REG_PS_THDL, data, 2) ;
    psthd = (data[1] << 8) | data[0] ;
    return( psthd ) ;
}

void     VCNL4100::setPsThd(uint16_t psthd) 
{
    uint8_t data[3] ;
    data[0] = REG_PS_THDL ;
    data[1] = psthd & 0xFF ;
    data[2] = (psthd >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

uint8_t VCNL4100::getPsThdl(void)
{
    uint16_t thd16 ;
    uint8_t thdl ;
    thd16 = getPsThd() ;
    thdl = thd16 & 0xFF ;
    return( thdl ) ;
}

void VCNL4100::setPsThdl(uint8_t thdl)
{
    uint16_t thd16 ;
    uint8_t data[3] ;
    thd16 = getPsThd() ;
    data[0] = REG_PS_THDL ;
    data[1] = thdl ;
    data[2] = (thd16 >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

uint8_t VCNL4100::getPsThdh(void)
{
    uint16_t thd16 ;
    uint8_t thdh ;
    thd16 = getPsThd() ;
    thdh = (thd16 >> 8) & 0xFF ;
    return( thdh ) ;
}

void VCNL4100::setPsThdh(uint8_t thdh)
{
    uint16_t thd16 ;
    uint8_t data[3] ;
    thd16 = getPsThd() ;
    data[0] = REG_PS_THDL ;
    data[1] = thd16 & 0xFF ;
    data[2] = thdh ;
    writeRegs(data, 3) ;
}

uint8_t  VCNL4100::getPsData(void) 
{
    uint8_t data[2] ;
    readRegs(REG_PS_DATA, data, 2) ;
    return(data[0]) ;
}

uint16_t VCNL4100::getAlsData(void) 
{
    uint16_t alsdata ;
    uint8_t data[2] ;
    readRegs(REG_ASL_DATA, data, 2) ;
    alsdata = (data[1] << 8) | data[0] ;
    return( alsdata ) ;
}

uint8_t  VCNL4100::getIntFlag(void) 
{
    uint8_t data[2] ;
    readRegs(REG_INT_FLAG, data, 2) ;
    return( data[1] ) ;
}
