#ifndef _VCNL4100_H_
#define _VCNL4100_H_
/**
 * Vishay VCNL4100
 * High Sensitivity Long Distance Proximity and
 * Ambient Light Sensor with I2C Interface
 */
 /*
 * @code
* @endcode
*/
class VCNL4100
{
public:
  /**
  * VCNL4100 constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  VCNL4100(PinName sda, PinName scl, int addr);

  /**
  * VCNL4100 destructor
  */
  ~VCNL4100();
  
/**
 * get ALS configuration 
 * @param none
 * @return uint8_t ALS CONF
 *
 * @note bit[7:6] ALS_IT ASL integration time setting,
 * @note longer integration time has higher sensitivity 
 * @note 00=80ms, 01=160ms, 10=320ms, 11=640ms
 * @note -
 * @note bit[5:4] (reserved)
 * @note -
 * @note bit[3:2] ALS interrupt persistence setting
 * @note 00=1, 01=2, 10=4, 11=8
 * @note -
 * @note bit[1] ALS_INT_EN 0=ALS interrupt disable, 1=ALS interrupt enable
 * @note -
 * @note bit[0] ALS_SD 0=ALS power on, 1=ALS Shut Down
 */
uint8_t  getAlsConf(void) ;

/**
 * set ALS configuration
 * @param uint8_t conf, value to assign
 * @returns none
 */
void     setAlsConf(uint8_t conf) ;

/**
 * Get ALS high interrupt threshold 
 * @param none
 * @returns uint16_t ALS high interrupt threshold
 */
uint16_t getAlsThdh(void) ;

/**
 * Set ALS high interrupt threshold
 * @param uint16_t thdh, value to assign
 * @returns none
 */
void     setAlsThdh(uint16_t thdh) ;

/**
 * Get ALS low interrupt threshold
 * @param none
 * @returns uint16_t ALS low interrupt threshold
 */
uint16_t getAlsThdl(void) ;

/**
 * Set ALS low interrupt threshold
 * @param uint16_t thdl, value to assign
 * @returns none
 */
void     setAlsThdl(uint16_t thdl) ;

/**
 * Get PS CONF1 and PS CONF2
 * @param none
 * @returns uint16_t PS_CONF2 as MSB, PS_CONF1 as LSB
 */
uint16_t getPsConf12(void) ;

/**
 * Set PS CONF1 and PS CONF2
 * @param uint16_t PS_CONF2 as MSB, PS_CONF1 as LSB
 * @returns none
 */
void     setPsConf12(uint16_t conf12) ;

/**
 * get PS CONF1
 * @param none
 * @returns Register value of PS_CONF1
 *
 * @note bit[7:6] PS_Duty PS IRED on/off duty ratio setting
 * @note 00=1/5120, 01=1/640, 10=1/80, 11=1/20
 * @note -
 * @note bit[5:4] PS_IT PS integration time setting
 * @note 00=1T, 01=1.3T, 10=1.6T, 11=2T
 * @note -
 * @note bit[3:2] PS_PERS PS interrupt persistence setting
 * @note 00=1, 01=2, 10=3, 11=4
 * @note -
 * @note bit[1] (reserved)
 * @note -
 * @note bit[0] PS_SD 0=PS power on, 1=PS shut down
 */
uint8_t  getPsConf1(void) ;

/**
 * set PS_CONF1
 *
 * @param uint8_t conf1 value to set
 * @returns none
 */
void     setPsConf1(uint8_t conf1) ;

/**
 * get PS_CONF2
 *
 * @note bit[7:6] PS_ITB PS IT bank setting
 * @note 00=1/2T, 01=1T, 10=2T, 11=4T
 * @note -
 * @note bit[5:4] PS_GAIN
 * @note 00=1/4, 01=1/2, 10=1, 11=2
 * @note -
 * @note bit[3] (reserved)
 * @note -
 * @note bit[2] PS_SP_INT_EN 
 * @note 0 = disable INT function for PS enter/leave sunlight protection mode
 * @note 1 = issue INT while PS enter/leave sunlight protection mode.
 * @note  While PS enter sunlight protection mode, the PS output will keep 0xFF
 * @note -
 * @note bit[1] (reserved)
 * @note -
 * @note bit[0] PS_INT_EN
 * @note 0 = PS INT function disable, 1 = PS INT function enable
 */
uint8_t  getPsConf2(void) ;

/**
 * set PS_CONF2
 *
 * @param uint8_t conf2 value to set
 * @returns none
 */
void     setPsConf2(uint8_t conf2) ;

/**
 * get PS_CONF3 and PS_SPO
 *
 * @param none
 * @returns uint16_t PS_CONF3 as LSB, PS_SPO as MSB
 */
uint16_t getPsConf3Spo(void) ;

/**
 * set PS_CONF3 and PS_SPO
 * 
 * @param uint16_t con3spo PS_CONF3 as LSB, PS_SPO as MSB
 * @returns none
 */
void     setPsConf3Spo(uint16_t conf3spo) ;

/**
 * get PS_CONF3
 *
 * @param none
 * @returns uint8_t PS_CONF3
 *
 * @note bit[7:6] PS_AV 
 * @note 00=1/2, 01=1/4, 10=1/8, 11=1/16
 * @note -
 * @note bit[5] PS_AV_EN 
 * @note 0= PS average function disable, 1= PS average function enable
 * @note -
 * @note bit[4] (reserved)
 * @note -
 * @note bit[3] PS_AF
 * @note 0= active force mode disable (normal mode), 1= active force mode enable
 * @note -
 * @note bit[2] PS_TRIG
 * @note 0= no PS active force mode trigger, 1= trigger one time cycle
 * @note VCNL4100 output once cycle data every time host writes in "1" to sensor.
 * @note The state returns to "0" automatically.
 * @note -
 * @note bit[1] PS_MPULSE 0= disable, 1= enable
 * @note PS multi pulse mode setting; PS multi puse number set by PS_AV[1:0]
 * @note -
 * @note bit[0] (reserved)
 */
uint8_t  getPsConf3(void) ;

/**
 * set PS_CONF3
 *
 * @param uint8_t conf3 value to set
 * @returns none
 */
void     setPsConf3(uint8_t conf3) ;

/**
 * get PS_SPO
 *
 * @param none
 * @returns uint8_t PS_SPO
 *
 * @note Set initial value to "0xA0" or "0x20"
 * @note bit[7:0] PS_SPO
 * @note Set initial value = 0xA0 (PS_OUT = 0xFF while PS into sunlight protection)
 * @note Set initial value = 0x20 (PS_OUT = 0x00 while PS into sunlight protection)
 *
 */
uint8_t  getSpo(void) ;

/**
 * set PS_SPO
 * 
 * @param uint8_t spo value to set (0xA0 or 0x20)
 * @returns none
 */
void     setSpo(uint8_t spo) ;

/**
 * get PS_THDL as LSB and PS_THDH as MSB
 *
 * @param none
 * @return uint16_t PS_THDL as LSB, PS_THDH as MSB
 */
uint16_t getPsThd(void) ;

/**
 * set PS_THDL as LSB and PS_THDH as MSB
 *
 * @param uint16_t PS_THDL as LSB, PS_THDH as MSB
 * @returns none
 */
void     setPsThd(uint16_t psthd) ;

/**
 * get PS_THDL
 * 
 * @param none
 * @returns uint8_t PS_THDL
 *
 * @note bit[7:0] PS_THDL 0x00 to 0xFF, PS low interrupt threshold setting
 */
uint8_t getPsThdl(void) ;

/**
 * set PS_THDL
 *
 * @param uint8_t thdl value to set
 * @returns none
 */
void setPsThdl(uint8_t thdl) ;

/**
 * get PS_THDH
 *
 * @param none
 * @returns uint8_t PS_THDH
 *
 * @note bit[7:0] PS_THDH 0x00 to 0xFF, PS high interrupt threshold setting
 */
 uint8_t getPsThdh(void) ;
 
 /**
  * set PS_THDH
  *
  * @param uint8_t thdh value to set
  * @returns none
  */
void setPsThdh(uint8_t thdh) ;

/**
 * get PS_DATA
 *
 * @param none
 * @returns uint8_t PS_DATA
 *
 * @note bit[7:0] 0x00 to 0xFF, PS output data
 */
uint8_t  getPsData(void) ;

/**
 * get ALS output data
 *
 * @param none
 * @returns uint16_t ALS output data
 */
uint16_t getAlsData(void) ;

/**
 * get INT_FLAG
 *
 * @param none
 * @returns uint8_t INT_FLAG
 *
 * @note bit[7] PS_SPF_LEAVE, PS leaving protection mode
 * @note bit[6] PS_SPF_ENTER, PS entering protection mode
 * @note bit[5] ALS_IF_L, ALS crossing low THD INT trigger event
 * @note bit[4] ALS_IF_H, ALS crossing high THD INT trigger event
 * @note bit[3:2] (reserved)
 * @note bit[1] PS_IF_CLOSE, PS rise above PS_THDH INT trigger event
 * @note bit[0] PS_IF_AWAY, PS drop below PS_THDL INT trigger event
 */
uint8_t  getIntFlag(void) ;
    
private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);

};

#endif /* _VCNL4100_H_ */