#ifndef _VCNL4020_H_
#define _VCNL4020_H_
/**
 * VCNL4020
 * Vishay VCNL4020 Fully Integrated Proximity 
 * and Ambient Light Sensor with Infrared Emitter,
 * I2C Interface, and Interrupt Function
 *
 * @code
#include "mbed.h"
#include "VCNL4020.h"
#define MSU_VCNL4020_I2C_ADDRESS (0x13)

#if defined (TARGET_KL25Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL46Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_K64F)
#define PIN_SCL  PTE24
#define PIN_SDA  PTE25
#elif defined (TARGET_K22F)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL05Z)
#define PIN_SCL  PTB3
#define PIN_SDA  PTB4
#elif defined (TARGET_NUCLEO_F411RE)
#define PIN_SCL  PB_8
#define PIN_SDA  PB_9
#else
 #error TARGET NOT DEFINED
#endif

#ifndef MSU_VCNL4020_I2C_ADDRESS
#define MSU_VCNL4020_I2C_ADDRESS 0x13
#endif

VCNL4020 *vcnl4020 = 0 ;

int main() {
    uint16_t als, prox ;

    vcnl4020 = new VCNL4020(PIN_SDA, PIN_SCL, MSU_VCNL4020_I2C_ADDRESS) ; 
    vcnl4020->setIrLedCurrent(1) ; // 10mA for LED 
    vcnl4020->disableAlsPeriodic() ;
    vcnl4020->disableProxPeriodic() ;
    vcnl4020->disableSelfTimed() ;
    
    printf("=== test VCNL4020 %s ===\n",  __DATE__) ;
    printf("Product ID = %d, Revision = %d\n", 
        vcnl4020->getProdID(),
        vcnl4020->getRevID()) ;
        
    while(1) {
        vcnl4020->trigBothOd() ; // trigger both ALS and PROX 
        while(vcnl4020->alsDataReady() == 0) {
            // wait for ambient light data ready 
        }
        als = vcnl4020->getAls() ;
        while(vcnl4020->proxDataReady() == 0) {
            // wait for proximate data ready 
        }
        prox = vcnl4020->getProx() ;
        printf("VCNL4020 Ambient Light: %d  Proximity: %d\n", als, prox) ;
        
        wait(1) ;
    }
}
* @endcode
*/
class VCNL4020
{
public:
  /**
  * VCNL4020 constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  VCNL4020(PinName sda, PinName scl, int addr);

  /**
  * VCNL4020 destructor
  */
  ~VCNL4020();

uint8_t getCommandReg(void) ;
void    setCommandReg(uint8_t cmd) ;
uint8_t getConfigLock(void) ;
void    setConfigLock(void) ;
void    clearConfigLock(void) ;
uint8_t alsDataReady(void) ;
uint16_t getAls(void) ;
uint8_t proxDataReady(void) ;
uint16_t getProx(void) ;
void    trigAlsOd(void) ;
void    trigProxOd(void) ;
void    trigBothOd(void) ;
uint8_t getAlsPeriodicEnable(void) ;
void    enableAlsPeriodic(void) ;
void    disableAlsPeriodic(void) ;
uint8_t getProxPeriodicEnable(void) ;
void    enableProxPeriodic(void) ;
void    disableProxPeriodic(void) ;
uint8_t getSelfTimedEnable(void) ;
void    enableSelfTimed(void) ;
void    disableSelfTimed(void) ;
uint8_t getFuseProgID(void) ;
uint8_t getIrLedCurrent(void) ;
void    setIrLedCurrent(uint8_t value) ;
uint8_t getIntControl(void) ;
void    setIntControl(uint8_t ctrl) ;
uint16_t getLowThreshold(void) ;
void     setLowThreshold(uint16_t thr) ;
uint16_t getHighThreshold(void) ;
void     setHighThreshold(uint16_t thr) ;
uint8_t getIntStatus(void) ;
uint8_t getProxModTiming(void) ;
void    setProxModTiming(uint8_t pmta) ;
uint8_t getProxRate(void) ;
void    setProxRate(uint8_t rate) ;
uint8_t getProdID(void) ;
uint8_t getRevID(void) ;
uint16_t getALS(void) ;

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);

};

#endif /* _VCNL4020_H_ */