/* Copyright (c) 2013-2015 Design Methodology Lab
*/
/*
* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef MMA8452Q_H
#define MMA8452Q_H

#include "mbed.h"

/**
* MMA8452Q accelerometer example
*
* @code
#include "mbed.h"
#include "MMA8452Q.h"

#if defined (TARGET_KL25Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL46Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_K64F)
#define PIN_SCL  PTE24
#define PIN_SDA  PTE25
#elif defined (TARGET_K22F)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL05Z)
#define PIN_SCL  PTB3
#define PIN_SDA  PTB4
#elif define (TARGET_F411RE)
#define PIN_SCL  PB_8
#define PIN_SDA  PB_9
#else
 #error TARGET NOT DEFINED
#endif
 
#define MMA8452_I2C_ADDRESS (0x1d)
 
int main(void) {
     float x, y, z ;

     MMA8452Q acc(PIN_SDA, PIN_SCL, MMA8452_I2C_ADDRESS);
 
     while (true) {       
         x = acc.getAccX() ;
         y = acc.getAccY() ;
         z = acc.getAccZ() ;
         printf("X[%.2f] Y[%.2f] Z[%.2f]\n",x, y, z) ;
         wait(0.1);
    }
}
* @endcode
*/
class MMA8452Q
{
public:
  /**
  * MMA8452Q constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  MMA8452Q(PinName sda, PinName scl, int addr);

  /**
  * MMA8452Q destructor
  */
  ~MMA8452Q();

  /**
   * Get the value of the WHO_AM_I register
   *
   * @returns WHO_AM_I value
   */
  uint8_t getWhoAmI();

  /**
   * Get X axis acceleration
   *
   * @returns X axis acceleration
   */
  float getAccX();

  /**
   * Get Y axis acceleration
   *
   * @returns Y axis acceleration
   */
  float getAccY();

  /**
   * Get Z axis acceleration
   *
   * @returns Z axis acceleration
   */
  float getAccZ();

  /**
   * Get XYZ axis acceleration
   *
   * @param res array where acceleration data will be stored
   */
  void getAccAllAxis(float * res);

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);
  int16_t getAccAxis(uint8_t addr);

};

#endif
