/**
 * MAX44005 RGB Color, Temperature,
 * and Infrared Proximity Sensor
 * I2C 7bit address: 0x4A
 */
#include "MAX44005.h"

/* some definitions here */
#define REG_INT_STATUS    0x00
#define REG_MAIN_CONFIG   0x01
#define REG_AMB_CONFIG    0x02
#define REG_PRX_CONFIG    0x03

/* AMBIENT + PROXIMITY READING */
#define REG_AMB_CLR_MSB   0x04
#define REG_AMB_CLR_LSB   0x05
#define REG_AMB_RED_MSB   0x06
#define REG_AMB_REG_LSB   0x07
#define REG_AMB_GRN_MSB   0x08
#define REG_AMB_GRN_LSB   0x09
#define REG_AMB_BLU_MSB   0x0A
#define REG_AMB_BLU_LSB   0x0B
#define REG_AMB_IR_MSB    0x0C
#define REG_AMB_IR_LSB    0x0D
#define REG_AMB_IRCMP_MSB 0x0E
#define REG_AMB_IRCMP_LSB 0x0F
#define REG_PRX_MSB       0x10
#define REG_PRX_LSB       0x11

/* TEMPERATURE SENSOR */
#define REG_TMP_MSB       0x12
#define REG_TMP_LSB       0x13

/* INTERRUPT THRESHOLDS */
#define REG_AMB_UTHR_MSB  0x14
#define REG_AMB_UTHR_LSB  0x15
#define REG_AMB_LTHR_MSB  0x16
#define REG_AMB_LTHR_LSB  0x17
#define REG_THR_PST       0x18
#define REG_PRX_UTHR_MSB  0x19
#define REG_PRX_UTHR_LSB  0x1A
#define REG_PRX_LTHR_MSB  0x1B
#define REG_PRX_LTHR_LSB  0x1C

/* AMBIENT ADC GAINS */
#define REG_TRIM_GAIN_CLR  0x1D
#define REG_TRIM_GAIN_RED  0x1E
#define REG_TRIM_GAIN_GRN  0x1F
#define REG_TRIM_GAIN_BLU  0x20
#define REG_TRIM_GAIN_IR   0x21

MAX44005::MAX44005(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr<<1) {
    // activate the peripheral
}

MAX44005::~MAX44005() { }

void MAX44005::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr} ;
    m_i2c.write(m_addr, t, 1, true) ;
    m_i2c.read(m_addr, (char*)data, len) ;
}

void MAX44005::writeRegs(uint8_t * data, int len) {
   m_i2c.write(m_addr, (char *)data, len) ;
}

int16_t MAX44005::getAMB_CLEAR(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_AMB_CLR_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}

int16_t MAX44005::getAMB_RED(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_AMB_RED_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}
    
int16_t MAX44005::getAMB_GREEN(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_AMB_GRN_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}
    
int16_t MAX44005::getAMB_BLUE(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_AMB_BLU_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}
 
int16_t MAX44005::getIR(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_AMB_IR_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}

int16_t MAX44005::getIRCOMP(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_AMB_IRCMP_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}

int16_t MAX44005::getTEMP(void) {
    int16_t value;
    uint8_t res[2];
    readRegs(REG_TMP_MSB, res, 2) ;
    value = (res[0] << 8)+res[1] ;
    return( value ) ;
}    

void MAX44005::enableTEMP(void) {
    uint8_t val[2] ;
    val[0] = REG_AMB_CONFIG ;
    readRegs(REG_AMB_CONFIG, &val[1], 1) ;
    val[1] = val[1] | 0x20 ; // set TEMPEN 
    writeRegs(val, 2) ;
}

void MAX44005::disableTEMP(void) {
    uint8_t val[2] ;
    val[0] = REG_AMB_CONFIG ;
    readRegs(REG_AMB_CONFIG, &val[1], 1) ;
    val[1] = val[1] & (~0x20) ; // clear TEMPEN 
    writeRegs(val, 2) ;
}

void MAX44005::enableAMBINT(void)
{
    uint8_t val[2] ;
    val[0] = REG_MAIN_CONFIG ;
    readRegs(REG_MAIN_CONFIG, &val[1], 1) ;
    val[1] |= 0x01 ;
    writeRegs(val, 2) ;
}

void MAX44005::disableAMBINT(void)
{
    uint8_t val[2] ;
    val[0] = REG_MAIN_CONFIG ;
    readRegs(REG_MAIN_CONFIG, &val[1], 1) ;
    val[1] &= (~0x01) ;
    writeRegs(val, 2) ;
}

void MAX44005::enablePRXINT(void)
{
    uint8_t val[2] ;
    val[0] = REG_MAIN_CONFIG ;
    readRegs(REG_MAIN_CONFIG, &val[1], 1) ;
    val[1] |= 0x02 ;
    writeRegs(val, 2) ;
}

void MAX44005::disablePRXINT(void) 
{
    uint8_t val[2] ;
    val[0] = REG_MAIN_CONFIG ;
    readRegs(REG_MAIN_CONFIG, &val[1], 1) ;
    val[1] &= (~0x02) ;
    writeRegs(val, 2) ;
}

void MAX44005::setMode(uint8_t newMode)
{
    uint8_t val[2] ;
    val[0] = REG_MAIN_CONFIG ;
    readRegs(REG_MAIN_CONFIG, &val[1], 1) ;
    val[1] = (0x70 & (newMode << 4)) | (val[1] & 0x0F)  ;
    writeRegs(val, 2) ;
}