/** MAX44000 Ambient and Infrared Proximity Sensor
 * I2C 7bit address: 0x4A
 *
 */

#ifndef _MAX44000_H_
#define _MAX44000_H_

#include "mbed.h"

/** @code 
 * Registers
 * addr : R/W : Rst  : Register
 * 0x00 : R   : 0x04 : Interrupt Status 
 * 0x01 : R/W : 0x24 : Main Configuration 
 * 0x02 : R/W : 0x00 : Receive Configuration 
 * 0x03 : R/W : 0x00 : Transmit Configuration 
 * 0x04 : R   : 0x00 : ADC High Byte (ALS) 
 * 0x05 : R   : 0x00 : ADC Low Byte (ALS) 
 * 0x16 : R   : 0x00 : ADC Byte (PROX) 
 * 0x06 : R/W : 0x00 : ALS Upper Threshold (High Byte) 
 * 0x07 : R/W : 0x00 : ALS Upper Threshold (Low Byte) 
 * 0x08 : R/W : 0x00 : ALS Lower Threshold (High Byte) 
 * 0x09 : R/W : 0x00 : ALS Lower Threshold (Low Byte) 
 * 0x0A : R/W : 0x00 : Threshold Persist Timer 
 * 0x0B : R/W : 0x00 : PROX Threshold Indicator 
 * 0x0C : R/W : 0x00 : PROX Threshold 
 * 0x0F : R/W : 0x80 : Digital Gain Trim of Green Channel 
 * 0x10 : R/W : 0x80 : Digital Gain Trim of Infrared Channel 
 * 
 * 0x00 : R   : 0x04 : Interrupt Status 
 * Bit[7:3] : (reserved)
 * Bit[2]   : PWRON : if set, indicates that a power-on-reset (POR) condition has occurred,
 *                    and any user-programmed thresholds cannot be valid anymore.
 * Bit[1]   : PRXINTS : indicates that an ambient light nterrupt condition occurred.
 * BIt[0]   : ALSINTS : indicates that a proximity receive interrupt condition occurred.
 *
 * 0x01 : R/W : 0x24 : Main Configuration 
 * Bit[7:6] : (reserved)
 * Bit[5]   : TRIM 
 *   0: Use bytes written to TRIM_GAIN_GREEN[7:0] and TRIM_GAIN_IR[7:0] registers
 *      to set the fine-trim gain of the green and IR gain channels.
 *   1: Use factory-programmed gains for green and IR channels.
 *      Ignore bytes written to TRIM_GAIN_GEEN[7:0] and TRIM_GAIN_IR[7:0] registers.
 * Bit[4:2] : mode
 *   000: Shutdown : Analog circuits are shut down, but the digital register retains values.
 *   001: ALS G-IR : Standard ALS mode stores the difference between green and infrared channel readings.
 *                   Proximity channel operation and updates are disabled.
 *   010: ALS G    : ALS green channel only. Proximity channel operation and updates are disabled.
 *   011: ALS IR   : Infrared channel only. Proximity channel operation and updates are disabled.
 *   100: ALS/PROX : ALS and PROX are interleaved continuously.
 *   101: PROX Only : PROX only continuously. ALS channel operation and updates are disabled.
 *   110: (reserved)
 *   111: (reserved)
 * Bit[1]   : PRXINTE : 
 *   0 : The PRXINTS bit remains unasserted, and proximity channel readings 
 *       are not compared with interrupt thresholds.
 *   1 : Detection of a proxmity interrupt event triggers a hardware interrupt (/INT pin is pulled low)
 *       and sets the PRXINTS bit (register 0x00, B1). Proximity channel readings are compared with
 *       proximity interrupt threshold settings and proximity persist timer.
 * Bit[0]   : ALSINTE
 *   0 : The ALSINTS bit reamins unasserted, and ALS channel readings are not compared with interrupt thresholds.
 *   1 : Detection of an ambient light interrupt event triggers a hardware interrupt (/INT pin is pullued low)
 *       and sets the ALSINTS bit (register 0x00, B0). ALS channel readings are compared with ALS interrupt 
 *       threshold settings and ALS persist timer.
 *
 * 0x02 : R/W : 0x00 : Receive Configuration 
 * Bit[7:4] : (reserved)
 * Bit[3:2] : ALSTIM : Ambient ADC Conversion Time
 *       | Intergration Time (ms) | Full-Scale ADC counts | Bit Resolution | Relative LSB Size |
 *    00 |        100             |        16,384         |        14      |           1x      | 
 *    01 |         25             |         4,096         |        12      |           4x      |
 *    10 |          6.25          |         1,024         |        10      |          16x      |
 *    11 |          1.5625        |           256         |         8      |          64x      |
 * Bit[1:0] : ALSPGA : Ambient Light Measurement Gain
 *       |        LUX/LSB         |   Relative LSB Size   |
 *    00 |        0.03125         |          1x           |
 *    01 |        0.125           |          4x           |
 *    10 |        0.5             |         16x           |
 *    11 |        4               |        128x           |
 *
 * 0x03 | R/W | 0x00 | Transmit Configuration 
 * 0x04 | R   | 0x00 | ADC High Byte (ALS) 
 * 0x05 | R   | 0x00 | ADC Low Byte (ALS) 
 * 0x16 | R   | 0x00 | ADC Byte (PROX) 
@endcode
*/
 


class MAX44000 
{
public:
 /**
 *  constructor
 *
 * @param sda SDA pin
 * @param scl SCL pin
 * @param addr address of the I2C peripheral
 */
 MAX44000(PinName sda, PinName scl, int addr) ;
 
 /**
  * destructor
  */
 ~MAX44000() ;

 /*
  * some member functions here (yet to be written)
  */

/**
 * Get Interrupt Status
 * @param uint8_t *data pointer to receive the 8bit register value.
 */
void getIntStatus(uint8_t *data) ;
/**
 * Get Main Configuration
 * @param uint8_t *conf pointer to receive the 8bit register value.
 */
void getMainConfig(uint8_t *conf) ;
/**
 * Set Mein Configuration
 * @param conf uint8_t to specify the 8bit register value.
 */
void setMainConfig(uint8_t newConf) ;
void getRxConfig(uint8_t *conf) ;
void setRxConfig(uint8_t newConf) ;
void getTxConfig(uint8_t *conf) ;
void setTxConfig(uint8_t newConf) ;
/**
 * Get ALS
 *
 * @param uint8_t *ofl overflow indicator
 * @param uint16_t *value ALS value
 */
void getALS(uint8_t *ofl, uint16_t *value) ;
/**
 * Get PROX
 *
 * @param uint8_t *value PROX value
 */
void getProx(uint8_t *value) ;
/**
 * Get ADC value(s)
 *
 * @param uint8_t *ofl overfolow indicator
 * @param uint16_t *als ALS data
 * @param uint8_t *prox PROX data
 */
void getADC(uint8_t *ofl, uint16_t *als, uint8_t *prox) ;
void getUPTHR(uint16_t *thr) ;
void setUPTHR(uint16_t newThr) ;
void getLOTHR(uint16_t *thr) ;
void setLOTHR(uint16_t newThr) ;
void getTHRPT(uint8_t *data) ;
void setTHRPT(uint8_t newData) ;
void getPRXTHR_IND(uint8_t *value) ;
void setPRXTHR_IND(uint8_t newValue) ;
void getPRX_THR(uint8_t *data) ;
void setPRX_THR(uint8_t newValue) ;
void getTRM_GAING(uint8_t *data) ;
void setTRM_GAING(uint8_t newValue) ;
void getTRM_GAINIR(uint8_t *data) ;
void setTRM_GAINIR(uint8_t newValue) ;

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t *data, int len) ;
  void writeRegs(uint8_t *data, int len) ;
} ;
#endif /* _MAX44000_H_ */
