#ifndef _MAX30101_H_
#define _MAX30101_H_
/**
 * MAX30101
 * High-Sensitivity Pulse Oximeter and
 * Heart-Rate Sensor for Wearable Health
 */

class MAX30101
{
public:
 /**
 * MAX30101 constructor
 *
 * @param sda SDA pin
 * @param scl SCL pin
 * @param addr 7bit address of the I2C peripheral
 */
 MAX30101(PinName sda, PinName scl, int addr) ;
 
 /**
  * MAX30101 destructor
  */
 ~MAX30101() ;

/**
 * Get part id
 * @returns PART_ID
 */
uint8_t  getID(void) ;

/**
 * Get revision id
 * @returns REV_ID
 */
uint8_t  getRev(void) ;

/**
 * Get interrupt status
 * @returns interrupt Status 1 as MSB and interrupt Status 2 as LSB
 */
uint16_t getIntStatus(void) ;

/**
 * Get interrupt enable
 * @returns interrupt Enable 1 as MSB and interrupt Status 2 as LSB
 */
uint16_t getIntEnable(void) ;

/**
 * Set interrupt enable
 * @param mask uint16_t MSB for Enable 1 and LSB for Enable 2
 */
void     setIntEnable(uint16_t mask) ;

/**
 * Get FIFO Write Pointer
 * @returns FIFO_WR_PTR
 */
uint8_t  getFIFO_WR_PTR(void) ;

/**
 * Set FIFO Write Pointer
 * @param data uint8_t FIFO_WR_PTR[4:0]
 */
void     setFIFO_WR_PTR(uint8_t data) ;

/**
 * Get FIFO Overflow Counter
 * @returns OVF_COUNTER
 */
uint8_t  getOVF_COUNTER(void) ;

/**
 * Set FIFO Overflow Counter
 * @param data uint8_t OVF_COUNTER[4:0]
 */
void     setOVF_COUNTER(uint8_t data) ;

/**
 * Get FIFO Read Pointer
 * @returns FIFO_RD_PTR
 */
uint8_t  getFIFO_RD_PTR(void) ;

/**
 * Set FIFO Read Pointer
 * @param data uint8_t FIFO_RD_PTR[4:0]
 */
void     setFIFO_RD_PTR(uint8_t data) ;

/**
 * Get FIFO Data
 * @returns FIFO_DATA 
 */
uint8_t  getFIFO_DATA(void) ;

/**
 * Set FIFO Data
 * @param FIFO_DATA[7:0]
 */
void     setFIFO_DATA(uint8_t data) ;

/** 
 * Get FIFO Confuguraion 
 * @returns FIFO_CONFIG
 */
uint8_t  getFIFO_CONFIG(void) ;

/**
 * Set FIFO Configuration
 * @param data uint8_t
 * @note bit[7:5] -> SMP_AVE[2:0]
 * @note bit[4]   -> FIFO_ROLLOVER_EN
 * @note bit[3:0] -> FIFO_A_FULL[3:0]
 */
void     setFIFO_CONFIG(uint8_t data) ;

/**
 * Get MODE Configuration
 * @returns MODE_CONFIG
 */
uint8_t  getMODE_CONFIG(void) ;

/**
 * Set MODE Configuration
 * @param data uint8_t
 * @note bit[7] Shutdown Control (SHDN)
 * @note bit[6] Reset Control (RESET)
 * @note bit[5:3] (reserved)
 * @note bit[2:0] Mode Control (MODE)
 */
void     setMODE_CONFIG(uint8_t data) ;

/**
 * Get SpO2 Configuration
 * @returns SPO2_CONFIG
 */
uint8_t  getSPO2_CONFIG(void) ;

/**
 * Set SpO2 Configuration
 * @param data uint8_t
 * @note bit[7] (reserved)
 * @note bit[6:5] SpO2 ADC Range Control (SPO2_ADC_RGE[1:0])
 * @note bit[4:2] SpO2 Sample Rate Control (SPO2_SR[2:0])
 * @note bit[1:0] LED Pulse Width Control (LED_PW[1:0])
 */
void     setSPO2_CONFIG(uint8_t data) ;

/**
 * Get LED1 Pulse Amplitude 
 * @returns LED1_PA
 */
uint8_t  getLED1_PA(void) ;

/**
 * Set LED1 Pulse Amplitude
 * @param data uint8_t LED1_PA[7:0]
 */
void     setLED1_PA(uint8_t data) ;

/**
 * Get LED2 Pulse Amplitude
 * @returns LED2_PA
 */
uint8_t  getLED2_PA(void) ;

/**
 * Set LED2 Pulse Amplitude
 * @param data uint8_t LED2_PA[7:0]
 */
void     setLED2_PA(uint8_t data) ;

/**
 * Get LED3 Pulse Amplitude
 * @returns LED3_PA
 */
uint8_t  getLED3_PA(void) ;

/**
 * Set LED3 Pulse Amplitude
 * @param data uint8_t LED3_PA[7:0]
 */
void     setLED3_PA(uint8_t data) ;

/**
 * Get Proximity Mode LED Pulse Amplitude
 * @returns PILOT_PA
 */
uint8_t  getPILOT_PA(void) ;

/**
 * Set Proximity Mode LED Pulse Amplitude
 * @param data uint8_t PILOT_PA[7:0]
 */
void     setPILOT_PA(uint8_t data) ;

/**
 * Get Multi-LED Mode Control Registers
 * @returns SLOT2, SLOT1, SLOT4, SLOT3 combined as uint16_t
 */
uint16_t getSLOT(void) ;

/**
 * Set Multi-LED Mode Control Registers
 * @param data uint16_t 
 * @note bit[15]    (reserved)
 * @note bit[14:12] SLOT2[2:0]
 * @note bit[11]    (reserved)
 * @note bit[10:8]  SLOT1[2:0]
 * @note bit[7]     (reserved)
 * @note bit[6:4]   SLOT4[2:0]
 * @note bit[3]     (reserved)
 * @note bit[2:0]   SLOT3[2:0]
 */
void     setSLOT(uint16_t data) ;

/**
 * Get Temperature Data (Integer part)
 * @returns TINT
 */
uint8_t  getTEMP_INT(void) ;

/**
 * Get Temperature Data (Fraction part)
 * @returns TFRAC
 */
uint8_t  getTEMP_FRAC(void) ;

/**
 * Get Die Temperature Configure
 * @returns TEMP_EN
 */
uint8_t  getTEMP_EN(void) ; 

/**
 * Set Die Temperature Configure
 */
 void setTEMP_EN(void) ;

/**
 * Get Temperature
 * @returns temperature as a float
 */
float    getTEMP(void) ;

/**
 * Get Proximity Mode Interrupt Threshold
 * @returns PROX_INT_THR
 */
uint8_t  getPROX_INT_THR(void) ;

/**
 * Set Proximity Mode Interrupt Threshold
 * @param data uint8_t PROX_INT_THRESH[7:0]
 */
void     setPROX_INT_THR(uint8_t data) ;

/**
 * Clear FIFO
 */
void     clearFIFO(void) ;

/**
 * Read FIFO
 * @returns FIFO data as uint32_t
 * @note FIFO data is always a 3-bytes data
 * @note byte1[1:0] : FIFO_DATA[17]-FIFO_DATA[16]
 * @note byte2[7:0] : FIFO_DATA[15]-FIFO_DATA[8]
 * @note byte3[7:0] : FIFO_DATA[7]-FIFO_DATA[0]
 * @note The data is left aligned, so FIFO_DATA[17]
 * @note is always MSB, although the data length 
 * @note can be 18-bit ~ 15-bit 
 */ 
uint32_t readFIFO(void) ;

/**
 * Reset
 */
void     reset(void) ;

/*
 * some member functions here (yet to be written)
 */
  
private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t *data, int len) ;
  void writeRegs(uint8_t *data, int len) ;
} ;

/* bits of Interrupt Status register */
/* In this class Interrupt Status 1 and
 * Interrupt Status 2 are combined to be a uint16_t
 */
#define INT_ST_A_FULL       0x8000 /* FIFO Almost Full Flag */
#define INT_ST_PPG_RGY      0x4000 /* New FIFO Data Ready */
#define INT_ST_ALC_OVF      0x2000 /* Ambient Light Cancellation Overflow */
#define INT_ST_PROX_INT     0x1000 /* Proximity Threshold Triggered */
#define INT_ST_PWR_RDY      0x0100 /* Power Ready Flag */
#define INT_ST_DIE_TEMP_RDY 0x0002 /* Internal Temperature Ready Flag */
/* bits of Interrupt Enable register */
#define INT_EN_A_FULL       0x8000 /* FIFO Almost Full Flag */
#define INT_EN_PPG_RGY      0x4000 /* New FIFO Data Ready */
#define INT_EN_ALC_OVF      0x2000 /* Ambient Light Cancellation Overflow */
#define INT_EN_PROX_INT     0x1000 /* Proximity Threshold Triggered */
#define INT_EN_DIE_TEMP_RDY 0x0002 /* Internal Temperature Ready Flag */
/* Mode configuration bits */
#define MODE_HR               0x02 /* Heart Rate mode */
#define MODE_SPO2             0x03 /* SpO2 mode */
#define MODE_MULTI_LED        0x07 /* Multi-LED mode */
#define MODE_RESET            0x40 /* Reset Control, this bit is auto clear */
#define MODE_SHDN             0x80 /* Shutdown Control */


#endif /* _MAX30101_H_ */