#ifndef _IS31SE5000_h_
#define _IS31SE5000_h_
/**
 * IR31SE5000 : IR sensor for touchless motion and proximity
 * IS31SE5000.h
 *
 */

#include "mbed.h"

class IS31SE5000 
{
public:
 /**
 *  constructor
 *
 * @param sda SDA pin
 * @param scl SCL pin
 * @param addr 7bit address of the I2C peripheral
 */
 IS31SE5000(PinName sda, PinName scl, int addr) ;
 
 ~IS31SE5000() ;
 
 /**
  * get status register value
  *
  * @param none
  * @returns uint8_t status register value
  *
  * @note status register 0x00
  * @note bit[7:4] (reserved)
  * @note -
  * @note bit[3:2] PD
  * @note 01: Be off,  02: Be close, Others: No motion
  * @note -
  * @note bit[1:0] MD
  * @note 01: From PD2 to PD1, 10: From PD1 to PD2, Others: No motion
  */
 uint8_t getStatus(void) ;
 
 /**
  * get Proximity Detection Value
  * 
  * @param none
  * @returns uint8_t value of PD part of status register bit[3:2]
  */
 uint8_t getPD(void) ; 
 
 /**
  * get Mortion Detection Value
  * @param none
  * @returns uint8_t value of MD part of status register bit[1:0]
  */
 uint8_t getMD(void) ; 
 
/**
 * get shutdown register value
 *
 * @param none
 * @returns uint8_t shutdown register value
 *
 * @note shutdown register (0x01)
 * @note bit[7:1] (reserved)
 * @note bit[0] SSD
 * @note 0: Software shutdown mode, 1: Normal operation
 */
 uint8_t get_shutdown_reg(void) ;
 
 /**
  * activate the device
  *
  * @param none
  * @returns none
  */
 void activate(void) ;
 
 /**
  * (Software) shutdown the device
  *
  * @param none
  * @returns none
  */
 void shutdown(void) ;
 
/**
 * get Configuration Register value
 *
 * @param none
 * @returns uint8_t configuration register value
 *
 * @note Configuration Register (0x11)
 * @note bit[7:4] (reserved)
 * @note -
 * @note bit[3:1] EC Emitting Current Setting
 * @note 000: 400mA, 001: 280mA, 01x: 210mA, 1xx: 70mA
 * @note - 
 * @note bit[0] MODE Mode Selection
 * @note 0: Motion mode, 1: Proximity mode
 */
 uint8_t get_config_reg(void) ;

/**
 * get EC value of Configuration Register bit[3:1]
 *
 * @param none
 * @returns uint8_t EC part of Configuration Register bit[3:1]
 */
uint8_t getEC(void) ;

/**
 * set EC value to Configuration Register [3:1]
 *
 * @param uint8_t EC Value to assign, 000b ~ 1xxb
 * @returns none
 */
void setEC(uint8_t ec_value) ;
 
/**
 * get MODE value of Configuration Register bit[0]
 *
 * @param none
 * @returns uint8_t MODE value of configuration Register bit[0]
 */
uint8_t getMode(void) ;

/**
 * Set device mode to motion detection
 *
 * @param none
 * @returns none
 */
void motionMode(void) ;

/**
 * Set device mode to proximity detection
 *
 * @param none
 * @returns none
 */
void proximityMode(void) ;
 
  private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t *data, int len) ;
  void writeRegs(uint8_t *data, int len) ;
  
  uint8_t shutdown_reg ;
  uint8_t config_reg ;
} ;

#endif /* _IS31SE5000_h_ */