/** 
 * IS31SE5000 : IR sensor for touchless motion and proximity
 *
 */
#include "mbed.h"
#include "IS31SE5000.h"

#define REG_STATUS      0x00
#define REG_SHUTDOWN    0x01
#define REG_CONFIG      0x11

#define SSD_SHUTDOWN    0x00
#define SSD_NORMAL      0x01

#define EC_400MA        0x00
#define EC_280MA        0x01
#define EC_210MA        0x02
#define EC_70MA         0x04

#define PD_BE_OFF       0x01
#define PD_BE_CLOSE     0x02

#define MD_PD2_PD1      0x01
#define MD_PD1_PD2      0x02

#define MODE_MOTION     0x00
#define MODE_PROXIMITY  0x01

/*
 * Register Definitions
 *
 * Status Register 0x00 (Read Only)
 * Bit [7:4] reserved
 * Bit [3:2] PD
 * Bit [1:0] MD
 * The Status Register stores the motion 
 * information which detected by IS31SE5000
 * 
 * PD : Proximity Detection
 * 01 : Be off
 * 10 : Be close
 * Others : No motion
 *
 * MD : Motion Detection
 * 01 : From PD2 to PD1
 * 10 : From PD1 to PD2
 * Others : No motion
 *
 * Shutdown Register 0x01 
 * Bit [7:1] reserved
 * Bit [0]   SSD (default = 0)
 * The Shutdown Register sets software shutdown
 * mode of IS31SE5000.
 *
 * SSD : Software Shutdown Enable
 * 0   : Software shutdown mode (default)
 * 1   : Normal operation
 *
 * Configuration Register
 * Bit [7:4] Reserved
 * Bit [3:1] EC   (default = 000)
 * Bit [0]   MODE (default = 0)
 * The Configuration Register sets the operating function.
 * 
 * EC : Emitting Current Setting
 * (Adjusting detection distance, 
 *  refer to detail information in Data Sheet Page 10)
 * 000 : 400mA (default)
 * 001 : 280mA
 * 01x : 210mA
 * 1xx :  70mA
 *
 * MODE : Mode Selection
 * 0    : Motion mode (default)
 * 1    : Proximity mode
 */
 
IS31SE5000::IS31SE5000(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr<<1) {
    // activate the peripheral
    shutdown_reg = 0x00 ;
    config_reg = 0x00 ;
}

IS31SE5000::~IS31SE5000() { }

 /* for Status Register */
uint8_t IS31SE5000::getStatus(void)  /* returns status register */
{
    uint8_t status_reg = 0x00 ;
    readRegs(REG_STATUS, &status_reg, 1) ;
    return( status_reg ) ;
}

uint8_t IS31SE5000::getPD(void)  /* get Proximity Detection Value */
{
    uint8_t data ;
    data = getStatus() ;
    data = (data >> 2) & 0x03 ;
    return(data) ;
}

uint8_t IS31SE5000::getMD(void)  /* get Motion Detection Value */
{
    uint8_t data ;
    data = getStatus() ;
    data &= 0x03 ;
    return(data) ;
}

/* for Shutdown Register */
void IS31SE5000::activate(void) 
{
    uint8_t data[2] = { REG_SHUTDOWN, SSD_NORMAL } ;
    writeRegs(data, 2) ;
}

void IS31SE5000::shutdown(void)
{
    uint8_t data[2] = { REG_SHUTDOWN, SSD_SHUTDOWN } ;
    writeRegs(data, 2) ;
}
 
 /* for Configuration Register */
void IS31SE5000::setEC(uint8_t ec_value) 
{
    uint8_t data[2] ;
    data[0] = REG_CONFIG ;
    config_reg = ((ec_value << 1)&0x07) | (config_reg & 0x01) ;
    data[1] = config_reg ;
    writeRegs(data, 2) ;
    
}

void IS31SE5000::motionMode(void)
{
    uint8_t data[2] ;
    data[0] = REG_CONFIG ;
    config_reg &= ~(MODE_PROXIMITY) ;
    data[1] = config_reg ; 
    writeRegs(data, 2) ;
}

void IS31SE5000::proximityMode(void) 
{
    uint8_t data[2] ;
    data[0] = REG_CONFIG ;
    config_reg |= MODE_PROXIMITY ;
    data[1] = config_reg ; 
    writeRegs(data, 2) ;
}

uint8_t IS31SE5000::getEC(void) 
{
    return( (config_reg >> 1) & 0x07) ;
}

uint8_t IS31SE5000::getMode(void) 
{
    return( config_reg & 0x01 ) ;
}
 
 /* */
uint8_t IS31SE5000::get_shutdown_reg(void)
{
    return(shutdown_reg) ;
}

uint8_t IS31SE5000::get_config_reg(void) 
{
    return( config_reg ) ;
}

void IS31SE5000::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void IS31SE5000::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}