/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef HIH6130_H
#define HIH6130_H

#include "mbed.h"

/**
* Honeywell HumidIcon Digital Humidity/Temperature Sensors
*
* @code
#include "mbed.h"
#include "HIH6130.h"
#define HIH6130_I2C_ADDRESS (0x27)

#if defined (TARGET_KL25Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL46Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_K64F)
#define PIN_SCL  PTE24
#define PIN_SDA  PTE25
#elif defined (TARGET_K22F)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL05Z)
#define PIN_SCL  PTB3
#define PIN_SDA  PTB4
#elif defined (TARGET_NUCLEO_F411RE)
#define PIN_SCL  PB_8
#define PIN_SDA  PB_9
#else
 #error TARGET NOT DEFINED
#endif

int main() {
    uint16_t result = 0 ;
    float humidity = 0 ;
    float temperature = 0 ;
    HIH6130 hih(PIN_SDA, PIN_SCL, HIH6130_I2C_ADDRESS) ;
    
    while(1) {
        result = hih.getValue(&humidity, &temperature) ;
        printf("Temp %.1f C   Humidity %.1f %%\n", temperature, humidity) ;
        wait(1) ;
    }
}
* @endcode
*/
class HIH6130
{
public:
  /**
  * HIH6130 constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  HIH6130(PinName sda, PinName scl, int addr);

  /**
  * HIH6130 destructor
  */
  ~HIH6130();
  
  /**
   * Send MR (Measurement Request)
   *
   * @param none
   * @returns none
   * @note this wakes up the device and triggers a measurement cycle.
   */
  void measure(void) ;

  /**
   * get humidity and temperature
   *
   * @param float *humidity humidity in %
   * @param float *temperature temperature in degree Celsius
   * @returns uint16_t status
   *
   * @note status bit 
   * @note 00b: normal operation
   * @note 01b: stale data
   * @note 10b: device in Command Mode
   * @note 11b: diagnostic condition
   */
  uint16_t getValue(float *humidity, float *temperature) ;
  
  void start_CM(void) ;
  void start_NOM(void) ;

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);

};

#endif
