#include "mbed.h"
#include "DS1683.h"

/* ETC stands for Elapsed Time Counter */

/* Command Register */ 
#define REG_COMMAND 0x00
/* Status Register */
#define REG_STATUS  0x01
/* Password Entry Register 0x02 - 0x05 */
#define REG_PWE     0x02
/* Event Counter Register 0x08 - 0x09 */
#define REG_EVENT   0x08
/* ETC Register */
#define REG_ETC     0x0A
/* Event Counter Alarm Limit Register 0x10 - 0x11 */
#define REG_ECAL    0x10
/* ETC Alarm Limit Register 0x12 - 0x15 */
#define REG_ETCAL   0x12
/* Configuration Register */
#define REG_CONFIG  0x16
/* Password Value 0x1A - 0x1D */
#define REG_PWV     0x1A
/* User EEPROM 0x20 - 0x2F */
#define REG_USR     0x20

/* Command Register Bits */
/* bit[7:1] Reserved */
/* bit[0] CLR_ALM write 1 to unlatch the acvie ALARM output */
// Clear Alarm Bit. This bit reads as a 0. Writing this bit to a 1
// unlatches the active ALARM output, setting the ALARM pin to 
// its inactive state if the alarm condition is no longer present.
// If the alarm condition persists, the ALARM pin once again
// asserts to its active state.
#define CLR_ALM_BIT 0x01

/* Status Register Bits */
/* bit[7:3] Reserved */
/* bit[2] EVENT */
// This bit indicates the status of the EVENT pin's logic level,
// detected after the tG glitch filter time. 
#define EVENT_BIT 0x04

/* bit[1] EVENT AF */
// Default value = 0. If the value in the Event Counter SRAM value is
// greater than or equal to the Event Counter Alarm Limit value,
// then this bit is automatically set to a value of 1 to indecate 
// the ALARM event. When the EVENT SRAM Counter value is less than
// the Event Counter Alarm Limit, this bit automatically set to a value of 0,
// indicating that there is no EVENT alarm.
#define EVENT_AF_BIT 0x02

/* bit[0] ETC AF */
// Default value = 0. If the value in the ETC SRAM value is greater than 
// or equal to the ETC Alarm Limit value, then this bit is automatically 
// set to a value of 1 to indicate an ALARM event. When the ETC SRAM value
// is less than the ETC Alarm Limit, this bit automatically set to a value of 0,
// indicating that there is no ETC alarm.
#define ETC_AF_BIT 0x01

/* Configuration Register */
/* bit[7:3] Reserved */
/* bit[2] ETC ALARM EN */
// Default value = 0, which is disabled. When set to a 1, 
// and if the ETC register is equal to or greater than the ETC Alarm limit, 
// then this device triggers the ETC Alarm Flag (ETC AF), 
// and the ALARM pin goes to its active state.
#define ETC_ALARM_EN_BIT 0x04

/* bit[1] EVENT ALARM EN */
// Default value = 0, which is disabled. When set to a 1, and if the Event
// Counter register is equal to or greater than the Event Counter Alarm limit,
// then this device triggers the Event Count Alarm Flag (EVENT AF),
// and the ALARM pin goes to its active state.
#define EVENT_ALARM_EN_BIT 0x02

/* bit[0] ALARM POL */
// Default value = 0, which sets the ALARM output active low.
// When set to a 1, the ALARM output is active high.
#define ALRM_POL_BIT 0x01

/* Member Functions */

DS1683::DS1683(PinName sda, PinName scl, PinName eventpin, PinName alarmpin, int addr) : 
event(eventpin), alarm(alarmpin), m_i2c(sda, scl),  m_addr(addr<<1)
{
 //   m_i2c.frequency(100000) ;
    m_i2c.frequency(10000) ;
    setConfig(0x00) ; /* disable ETC_ALRM_EN and EVENT_ALRM_EN */
    alarmPol(0) ; /* Low Active */
//    alarmPol(1) ; /* Active High */    
}

DS1683::~DS1683() 
{ 
} 

int DS1683::read(int addr, uint8_t *data, int len) 
{
    int result ;
    result = readRegs(addr, data, len) ;
//    wait_ms(1000) ;
    wait(0.01) ;
    return( result ) ;
}

int DS1683::write(int addr, uint8_t *data, int len) 
{
    uint8_t *buf ;
    int ack ;
    buf = new uint8_t[len+1] ;
    buf[0] = addr ;
    for (int i = 0 ; i < len ; i++ ) {
        buf[i+1] = data[i] ;
    }
    ack = writeRegs(buf, len+1) ;
//    wait_ms(1000) ;
    wait(0.01) ;
    delete buf ;
    return( ack ) ;
}

int DS1683::readRegs(int addr, uint8_t * data, int len) {
    int result ;
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    result = m_i2c.read(m_addr, (char *)data, len);
    return( result ) ;
}

int DS1683::writeRegs(uint8_t * data, int len) {
    int ack ;
//    m_i2c.stop() ;
//    wait_ms(1000) ;
    wait(0.01) ;
    ack = m_i2c.write(m_addr, (char *)data, len);
//    m_i2c.stop() ;
    return( ack ) ;
}

uint8_t DS1683::readReg8(int addr)
{
    uint8_t data[1] ;
    readRegs(addr, data, 1) ;
    return( data[0] ) ;
}

void DS1683::writeReg8(int addr, uint8_t value)
{
    uint8_t data[2] ;
    data[0] = addr ;
    data[1] = value ;
    writeRegs(data, 2) ;
}

uint16_t DS1683::readReg16(int addr) 
{
    uint8_t data[2] ;
    uint16_t value = 0 ;
    readRegs(addr, data, 2) ;
    value = data[1] ;
    value = (value << 8) | data[1] ;
    return(value) ;
}

void DS1683::writeReg16(int addr, uint16_t value) 
{
    uint8_t data[3] ;
    data[0] = addr ;
    data[1] = value & 0xFF ;
    data[2] = (value >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

uint32_t DS1683::readReg32(int addr) 
{
    uint8_t data[4] ;
    uint32_t value = 0 ;
    readRegs(addr, data, 4) ;
    value = data[3] ;
    value = (value << 8) | data[2] ;
    value = (value << 8) | data[1] ;
    value = (value << 8) | data[0] ;
    return(value) ;
}

void DS1683::writeReg32(int addr, uint32_t value) 
{
    uint8_t data[5] ;
    data[0] = addr ;
    data[1] = value & 0xFF ;
    data[2] = (value >> 8) & 0xFF ;
    data[3] = (value >> 16) & 0xFF ;
    data[4] = (value >> 24) & 0xFF ;
    writeRegs(data, 5) ;
}

void DS1683::setConfig(uint8_t conf) 
{
    writeReg8(REG_CONFIG, conf) ;
}

uint8_t DS1683::getConfig(void)
{
    return( readReg8(REG_CONFIG) ) ;
}

uint8_t DS1683::getStatus(void)
{
    return(readReg8(REG_STATUS)) ;
}

void DS1683::setETC(uint32_t count)
{
    writeReg32(REG_ETC, count) ;
}

uint32_t DS1683::getETC(void)
{
    return(readReg32(REG_ETC)) ;
}

void DS1683::setEventCount(uint16_t count)
{
    writeReg16(REG_EVENT, count) ;
}

uint16_t DS1683::getEventCount(void)
{
    return(readReg16(REG_EVENT)) ;
}

void DS1683::clearRegs(void)
{
    writeReg16(REG_ECAL,  0x0000) ;
    writeReg16(REG_EVENT, 0x0000) ;
    writeReg32(REG_ETCAL, 0x00000000) ;
    writeReg32(REG_ETC,   0x00000000) ;
    writeReg8(REG_CONFIG, 0x00) ;
}

void DS1683::dumpRegs(void)
{
 //   uint8_t data[5] ;
//data[0] = 0 ;
//    readRegs(REG_COMMAND, data, 1) ;
    printf("Command: 0x%02X ", readReg8(REG_COMMAND)) ;
    printf("Status:  0x%02X ", getStatus()) ;
    printf("Config:  0x%02X\n", getConfig()) ;
}

void DS1683::enterPW(uint32_t pass)
{
    uint8_t data[5] ;
    data[0] = REG_PWE ; /* Password Entry */
    data[1] = pass & 0xFF ;
    data[2] = (pass >> 8) & 0xFF ;
    data[3] = (pass >> 16) & 0xFF ;
    data[4] = (pass >> 24) & 0xFF ;
    writeRegs(data, 5) ;
} 

void DS1683::dumpETC(void)
{
    uint32_t data[2] ;
    data[0] = getETC() ;
    data[1] = getETCAlarm() ;
    printf("ETC count: 0x%08X / limit: 0x%08X flag: ", data[0], data[1]) ;
    if (getStatus() & ETC_AF_BIT) {
        printf("ON") ;
    } else {
        printf("OFF") ;
    }
    printf("\n") ;
}

void DS1683::dumpEvent(void)
{
    uint16_t data[2] ;
    data[0] = getEventCount() ;
    data[1] = getEventAlarm() ;
    printf("Event Counter count: %04X / limit: %04X flag: ", data[0], data[1]) ;
    if (getStatus() & EVENT_AF_BIT) {
        printf("ON") ;
    } else {
        printf("OFF") ;
    }
    printf("\n") ;
}
    
void DS1683::setETCAlarm(uint32_t count)
{
    writeReg32(REG_ETCAL, count) ;
}

uint32_t DS1683::getETCAlarm(void)
{
    return(readReg32(REG_ETCAL)) ;
}

void DS1683::clearAlarm(void)
{
    uint8_t data[2] ;
    data[0] = REG_COMMAND ;
    data[1] = 0x01 ;
    writeRegs(data, 2) ;
}

void DS1683::clearEvent(void) 
{
    writeReg16(REG_EVENT, 0x0000) ;
}

void DS1683::setEventAlarm(uint16_t count)
{
    writeReg16(REG_ECAL, count) ;
}

uint16_t DS1683::getEventAlarm(void)
{
    return(readReg16(REG_ECAL)) ;
}

void DS1683::clearETC(void)
{
    writeReg32(REG_ETC, 0x00000000) ;
}
    
void DS1683::enableETCAlarm(void) 
{
    uint8_t config ;
    config = getConfig() ;
    config |= ETC_ALARM_EN_BIT ;
    setConfig(config) ;
}

void DS1683::disableETCAlarm(void) 
{
    uint8_t config ;
    config = getConfig() ;
    config &= ~ETC_ALARM_EN_BIT ;
    setConfig(config) ;
}

void DS1683::enableEventAlarm(void) 
{
    uint8_t config ;
    config = getConfig() ;
    config |= EVENT_ALARM_EN_BIT ;
    setConfig(config) ;
}

void DS1683::disableEventAlarm(void) 
{
    uint8_t config ;
    config = getConfig() ;
    config &= ~EVENT_ALARM_EN_BIT ;
    setConfig(config) ;
}

void DS1683::alarmPol(int pol)
{
    uint8_t config ;
    config = getConfig() ;
    if (pol) {
        config |= 0x01 ;
    } else {
        config &= ~0x01 ;
    }
    setConfig(config) ;
}