#include "mbed.h"
#include "AK9753.h"

#define REG_WIA1       0x00
#define REG_WIA2       0x01
#define REG_INFO1      0x02
#define REG_INFO2      0x03
#define REG_INTST      0x04
#define REG_ST1        0x05
#define REG_IR1L       0x06
#define REG_IR1H       0x07
#define REG_IR2L       0x08
#define REG_IR2H       0x09
#define REG_IR3L       0x0A
#define REG_IR3H       0x0B
#define REG_IR4L       0x0C
#define REG_IR4H       0x0D
#define REG_TMPL       0x0E
#define REG_TMPH       0x0F
#define REG_ST2        0x10
#define REG_ETH13H_LSB 0x11
#define REG_ETH13H_MSB 0x12
#define REG_ETH13L_LSB 0x13
#define REG_ETH13L_MSB 0x14
#define REG_ETH24H_LSB 0x15
#define REG_ETH24H_MSB 0x16
#define REG_ETH24L_LSB 0x17
#define REG_ETH24L_MSB 0x18
#define REG_EHYS13     0x19
#define REG_EHYS24     0x1A
#define REG_EINTEN     0x1B
#define REG_ECNTL1     0x1C
#define REG_CNTL2      0x1D

/* EEPROM */
#define REG_EKEY          0x50
#define EEPROM_ETH13H_LSB 0x51
#define EEPROM_ETH13H_MSB 0x52
#define EEPROM_ETH13L_LSB 0x53
#define EEPROM_ETH13L_MSB 0x54
#define EEPROM_ETH24H_LSB 0x55
#define EEPROM_ETH24H_MSB 0x56
#define EEPROM_ETH24L_LSB 0x57
#define EEPROM_ETH24L_MSB 0x58
#define EEPROM_EHYS13     0x59
#define EEPROM_EHYS24     0x5A
#define EEPROM_EINTEN     0x5B
#define EEPROM_ECNTL1     0x5C


AK9753::AK9753(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr<<1) {
    // activate the peripheral
}

AK9753::~AK9753() { }

int AK9753::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    int result ;
    result = m_i2c.write(m_addr, t, 1, true);
    if (result == 0) {
        result = m_i2c.read(m_addr, (char *)data, len);
    }
    return( result ) ;
}

int AK9753::writeRegs(uint8_t * data, int len) {
    int result ;
    result = m_i2c.write(m_addr, (char *)data, len);
    return( result ) ;
}

/**
 * getCompanyCode
 * the code is expected to be 0x48 
 */
uint8_t AK9753::getCompanyCode(void) 
{
    uint8_t data ;
    readRegs(REG_WIA1, &data, 1) ;
    return(data) ;
}

/**
 * getDeviceID
 * the ID is expected to be 0x13
 */
uint8_t AK9753::getDeviceID(void) 
{
    uint8_t data ;
    readRegs(REG_WIA2, &data, 1) ;
    return(data) ;
}

bool AK9753::dataReady(void) /* returns ST1[0], read ST2 to clear */
{
    uint8_t data ;
    readRegs(REG_ST1, &data, 1) ;
    return((data & 0x01) == 0x01) ;
}

bool AK9753::dataOverRun(void)
{
    uint8_t data ;
    readRegs(REG_ST2, &data, 1) ;
    return((data & 0x02) == 0x02) ;
}

uint8_t AK9753::getINTST(void)  /** return REG_INTST */
{
    uint8_t data ;
    readRegs(REG_INTST, &data, 1) ;
    return( data ) ;
}

uint8_t  AK9753::getST1(void) 
{
    uint8_t data ;
    readRegs(REG_ST1, &data, 1) ;
    return( data ) ;
}

int16_t AK9753::getRawIR1(void) 
{
    uint8_t data[2] ;
    int16_t IR ;
    readRegs(REG_IR1L, data, 2) ;
    IR = (data[1] << 8) | data[0] ;
    return( IR ) ;
}

float AK9753::getIR1(void) 
{
    int16_t iValue ;
    float fValue ;
    iValue = getRawIR1() ;
    fValue = 14286.8 * iValue / 32768.0 ;
    return( fValue ) ;
}

int16_t AK9753::getRawIR2(void) 
{    
    uint8_t data[2] ;
    int16_t IR ;
    readRegs(REG_IR2L, data, 2) ;
    IR = (data[1] << 8) | data[0] ;
    return( IR ) ;
}

float AK9753::getIR2(void) 
{
    int16_t iValue ;
    float fValue ;
    iValue = getRawIR2() ;
    fValue = 14286.8 * iValue / 32768.0 ;
    return( fValue ) ;
}

int16_t AK9753::getRawIR3(void) 
{
    uint8_t data[2] ;
    int16_t IR ;
    readRegs(REG_IR3L, data, 2) ;
    IR = (data[1] << 8) | data[0] ;
    return( IR ) ;
}

float AK9753::getIR3(void) 
{
    int16_t iValue ;
    float fValue ;
    iValue = getRawIR3() ;
    fValue = 14286.8 * iValue / 32768.0 ;
    return( fValue ) ;
}

int16_t AK9753::getRawIR4(void) 
{
    uint8_t data[2] ;
    int16_t IR ;
    readRegs(REG_IR4L, data, 2) ;
    IR = (data[1] << 8) | data[0] ;
    return( IR ) ;
}

float AK9753::getIR4(void) 
{
    int16_t iValue ;
    float fValue ;
    iValue = getRawIR4() ;
    fValue = 14286.8 * iValue / 32768.0 ;
    return( fValue ) ;
}

int16_t AK9753::getRawTMP(void) 
{
    uint8_t data[2] ;
    int16_t temp ;
    readRegs(REG_TMPL, data, 2) ;
    temp = (data[1] << 8) | data[0] ;
    return( temp ) ;
}

float AK9753::getTMP(void) 
{
    int16_t iValue ;
    float fValue ;
    iValue = getRawTMP() ;
    fValue = 26.75 +  (60.0 - 26.75) * iValue / 17792.0 ;
    return( fValue ) ;
}

uint8_t  AK9753::getST2(void) 
{
    uint8_t data ;
    readRegs(REG_ST2, &data, 1) ;
    return( data ) ;
}

int16_t AK9753::getETH13H(void)
{
    int16_t value ;
    uint8_t data[2] ;
    readRegs(REG_ETH13H_LSB, data, 2) ;
    value = (data[1] << 8) | data[0] ;
    return( value ) ;
}

int16_t AK9753::getETH13L(void)
{
    int16_t value ;
    uint8_t data[2] ;
    readRegs(REG_ETH13L_LSB, data, 2) ;
    value = (data[1] << 8) | data[0] ;
    return( value ) ;
}

int16_t AK9753::getETH24H(void)
{
    int16_t value ;
    uint8_t data[2] ;
    readRegs(REG_ETH24H_LSB, data, 2) ;
    value = (data[1] << 8) | data[0] ;
    return( value ) ;
}

int16_t AK9753::getETH24L(void)
{
    int16_t value ;
    uint8_t data[2] ;
    readRegs(REG_ETH24L_LSB, data, 2) ;
    value = (data[1] << 8) | data[0] ;
    return( value ) ;
}

uint8_t AK9753::getEHYS13(void)
{
    uint8_t data ;
    readRegs(REG_EHYS13, &data, 1) ;
    return( data ) ;
}

uint8_t AK9753::getEHYS24(void)
{
    uint8_t data ;
    readRegs(REG_EHYS24, &data, 1) ;
    return( data ) ;
}

uint8_t AK9753::getEINTEN(void)
{
    uint8_t data ;
    readRegs(REG_EINTEN, &data, 1) ;
    return( data ) ;   
}

uint8_t AK9753::getECNTL1(void)
{
    uint8_t data ;
    readRegs(REG_ECNTL1, &data, 1) ;
    return( data ) ;      
}

uint8_t AK9753::getCNTL2(void)
{
    uint8_t data ;
    readRegs(REG_CNTL2, &data, 1) ;
    return( data ) ;      
}

void AK9753::setETH13H(int16_t value) 
{
    uint8_t data[3] ;
    data[0] = REG_ETH13H_LSB ;
    data[1] = value & 0xFF ;
    data[2] = (value >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

void AK9753::setETH13L(int16_t value) 
{
    uint8_t data[3] ;
    data[0] = REG_ETH13L_LSB ;
    data[1] = value & 0xFF ;
    data[2] = (value >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

void AK9753::setETH24H(int16_t value) 
{
    uint8_t data[3] ;
    data[0] = REG_ETH24H_LSB ;
    data[1] = value & 0xFF ;
    data[2] = (value >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

void AK9753::setETH24L(int16_t value) 
{
    uint8_t data[3] ;
    data[0] = REG_ETH24L_LSB ;
    data[1] = value & 0xFF ;
    data[2] = (value >> 8) & 0xFF ;
    writeRegs(data, 3) ;
}

void AK9753::setEHYS13(uint8_t value) 
{
    uint8_t data[2] ;
    data[0] = REG_EHYS13 ;
    data[1] = value ;
    writeRegs(data, 2) ;
}

void AK9753::setEHYS24(uint8_t value) 
{
    uint8_t data[2] ;
    data[0] = REG_EHYS24 ;
    data[1] = value ;
    writeRegs(data, 2) ;    
}

void AK9753::setEINTEN(uint8_t value) 
{
    uint8_t data[2] ;
    data[0] = REG_EINTEN ;
    data[1] = value ;
    writeRegs(data, 2) ; 
}

void AK9753::setECNTL1(uint8_t value) 
{
    uint8_t data[2] ;
    data[0] = REG_ECNTL1 ;
    data[1] = value ;
    writeRegs(data, 2) ; 
}

void AK9753::software_reset(void) 
{
    uint8_t data[2] = { REG_CNTL2, 0xFF } ;
    writeRegs(data, 2) ;
}