#include "mbed.h"
#include "AK9752.h"

#define REG_WIA1       0x00
#define REG_WIA2       0x01
#define REG_INFO1      0x02
#define REG_INFO2      0x03
#define REG_ST1        0x04
#define REG_INTCAUSE   0x05
#define REG_IR_LSB     0x06
#define REG_IR_MSB     0x07
#define REG_TMP_LSB    0x08
#define REG_TMP_MSB    0x09
#define REG_ST2        0x0A
#define REG_THIRH_LSB  0x0B
#define REG_THIRH_MSB  0x0C
#define REG_THIRL_LSB  0x0D
#define REG_THIRL_MSB  0x0E
#define REG_THTMPH_LSB 0x0F
#define REG_THTMPH_MSB 0x10
#define REG_THTMPL_LSB 0x11
#define REG_THTMPL_MSB 0x12
#define REG_INTEN      0x13
#define REG_CNTL1      0x14
#define REG_CNTL2      0x15
#define REG_CNTL3      0x16


AK9752::AK9752(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr<<1) {
    // activate the peripheral
}

AK9752::~AK9752() { }

int AK9752::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    int result ;
    result = m_i2c.write(m_addr, t, 1, true);
    if (result == 0) {
        result = m_i2c.read(m_addr, (char *)data, len);
    }
    return( result ) ;
}

int AK9752::writeRegs(uint8_t * data, int len) {
    int result ;
    result = m_i2c.write(m_addr, (char *)data, len);
    return( result ) ;
}

/**
 * getCompanyCode
 * the code is expected to be 0x48 
 */
uint8_t AK9752::getCompanyCode(void) 
{
    uint8_t data ;
    readRegs(REG_WIA1, &data, 1) ;
    return(data) ;
}

/**
 * getDeviceID
 * the ID is expected to be 0x14
 */
uint8_t AK9752::getDeviceID(void) 
{
    uint8_t data ;
    readRegs(REG_WIA2, &data, 1) ;
    return(data) ;
}

bool    AK9752::dataReady(void) /* returns ST1[0], read ST2 to clear */
{
    uint8_t data ;
    readRegs(REG_ST1, &data, 1) ;
    return((data & 0x01) == 0x01) ;
}

uint8_t AK9752::getIntcause(void)  /* get REG_INTCAUSE (0x05) */
{
    uint8_t data ;
    readRegs(REG_INTCAUSE, &data, 1) ;
    return( data ) ;
}

int16_t AK9752::getRawIR(void)  /* raw data, must be mulitplied by 0.4578 for pA */
{
    uint16_t temp ;
    uint8_t data[2] ;
    readRegs(REG_IR_LSB, data, 2) ;
    temp = (data[1] << 8) | data[0] ;
    return( temp ) ;
}

float AK9752::getIR(void)
{
    float fIR ;
    uint16_t iIR ;
    iIR = getRawIR() ;
    fIR = iIR * 0.4578 ;
    return( fIR ) ;
}

int16_t AK9752::getRawTMP(void)  /* raw data, must be multiplied by 0.0019837 and + 25 */
{
    uint16_t temp ;
    uint8_t data[2] ;
    readRegs(REG_TMP_LSB, data, 2) ;
    temp = (data[1] << 8) | data[0] ;
    return( temp ) ;
}

float AK9752::getTMP(void)
{
    float ftemp ;
    uint16_t itemp ;
    itemp = getRawTMP() ;
    ftemp = itemp * 0.0019837 + 25.0 ;
    return( ftemp ) ;
}

bool    AK9752::dataOverRun(void)  /* check data over run and clear data ready */
{
    uint8_t data ;
    readRegs(REG_ST2, &data, 1) ;
    return((data & 0x01) == 0x01) ;
}

void    AK9752::getTHIR(int16_t  *high, int16_t *low) 
{
    uint8_t data[4] ;
    readRegs(REG_THIRH_LSB, data, 4) ;
    *high = (data[1] << 8) | data[0] ;
    *low  = (data[3] << 8) | data[2] ;
}

void    AK9752::setTHIR(int16_t   high, int16_t  low) 
{
    uint8_t data[5] ;
    data[0] = REG_THIRH_LSB ;
    data[1] = high & 0xFF ;
    data[2] = (high >> 8) & 0xFF ;
    data[3] = low & 0xFF ;
    data[4] = (low >> 8) & 0xFF ;
    writeRegs(data, 5) ;
}

void    AK9752::getTHTMP(int16_t *high, int16_t *low) 
{
    uint8_t data[4] ;
    readRegs(REG_THTMPH_LSB, data, 4) ;
    *high = (data[1] << 8) | data[0] ;
    *low  = (data[3] << 8) | data[2] ;
}

void    AK9752::setTHTMP(int16_t  high, int16_t  low) 
{
    uint8_t data[5] ;
    data[0] = REG_THTMPH_LSB ;
    data[1] = high & 0xFF ;
    data[2] = (high >> 8) & 0xFF ;
    data[3] = low & 0xFF ;
    data[4] = (low >> 8) & 0xFF ;
    writeRegs(data, 5) ;
}

/**
 * ITEN[4] = IRHI  : IR upper threshold interrupt enable
 * ITEN[3] = IRLI  : IR lower threshold interrupt enable
 * ITEN[2] = TMPHI : Temp upper threshold interrupt enable
 * ITEN[1] = TMPLI : Temp lower threshold interrupt enable
 * ITEN[0] = DRI   : Data ready interrupt enable
 */
uint8_t AK9752::getINTEN(void) 
{
    uint8_t value ;
    readRegs(REG_INTEN, &value, 1) ;
    return(value) ;
}

void AK9752::setINTEN(uint8_t value) 
{
    uint8_t t[2] ;
    t[0] = REG_INTEN ;
    t[1] = value ;
    writeRegs(t, 2) ;
}

/**
 * CNTL1
 * CNTL1[4:2] = FCTMP[2:0]
 * FCTMP = 000 : no digital filter
 * FCTMP = 001 : Fc = 2.5Hz
 * FCTMP = 010 : Fc = 0.9Hz
 * FCTMP = 011 : Fc = 0.45Hz
 * FCTMP = 100 : Fc = 0.22Hz
 * FCTMP = other : (reserved)
 * CNTL1[1:0] = FCIR[1:0]
 * FCIR = 00 : no digital filter
 * FCIR = 01 : Fc = 2.5Hz
 * FCIR = 10 : Fc = 0.9Hz
 * FCIR = 11 : Fc = 0.45Hz
 */
uint8_t AK9752::getCNTL1(void) 
{
    uint8_t value ;
    readRegs(REG_CNTL1, &value, 1) ;
    return(value) ;
}

void    AK9752::setCNTL1(uint8_t value) 
{
    uint8_t t[2] ;
    t[0] = REG_CNTL1 ;
    t[1] = value ;
    writeRegs(t, 2) ;
}

/**
 * CNTL2
 * CNTL2[1:0] = MODE[1:0]
 * MODE = 00 : standby
 * MODE = 01 : continuous measure
 * MODE = 10 : single shot
 * MODE = 11 : (reserved)
 */
uint8_t AK9752::getCNTL2(void) 
{
    uint8_t value ;
    readRegs(REG_CNTL2, &value, 1) ;
    return(value) ;
}

void    AK9752::setCNTL2(uint8_t value) 
{
    uint8_t t[2] ;
    t[0] = REG_CNTL2 ;
    t[1] = value ;
    writeRegs(t, 2) ;
}

void    AK9752::software_reset(void) 
{
    uint8_t t[2] = { REG_CNTL3, 0xFF } ;
    writeRegs(t, 2) ;
}