#include "mbed.h"
/*  This function demonstrates the use of the Ticker object to create a
periodic interrupt.  One common task for the Ticker would be to sample
analog data, process it, and put it out on an AnalogOut.  This code
applies two basic signal processing filters; an average filter and an echo
feedback filter.
*/

// The input signal will come in on A0.
AnalogIn sound_input(A0);

// The light will pulse along with the input signal (good for debugging).
PwmOut light(D12);

// The original and processed data will be output as analog signals.
AnalogOut original(D13);
AnalogOut processed(A2);

// The Ticker object creates a periodic interrupt.
Ticker sample;
// The sample_function will be called on the interrupt to get the new
// audio sample and process it.
void sample_function();

// Define arrays for past outputs and past inputs.  Outputs are used for echo
// (since it is a feedback process) and inputs are used for averaging filter.
float past_outputs[4000];
#define FILTER_SIZE 20
float past_inputs[FILTER_SIZE];

// Helpful math functions.
float average(float *array, int N);
float limit(float x);

int main()
{
    // Zero out the arrays.
    for(int k=0; k<4000; k++) {
        past_outputs[k]=0;
    }
    for(int k=0; k<FILTER_SIZE; k++) {
        past_inputs[k]=0;
    }
    
    // Attach the function that will handle samples to the Ticker interrupt.
    // Set it to go every 125 microseconds (8000 Hz).
    sample.attach_us(&sample_function,125);
 
    while(1);   
    
}

void sample_function() {
    
    //These keep track of the current sample.
    static int count=0;
    static float newsample=0;
    
    // First create the output based on previous data.
//    processed = average(past_inputs,FILTER_SIZE);
    processed=past_outputs[count];
    original=newsample;
    light=1-newsample;
    
    // Then sample the new data to use for next time (since the AD conversion
    // and computation can take a while)
    newsample=sound_input;
    count++;
    count%=4000;
    past_outputs[count] = limit(newsample + past_outputs[count]*.5);
//    count%=FILTER_SIZE;
//    past_inputs[count] = newsample;

}

// Helpful function to max out the sample at 1.0.
float limit(float x) {
    if(x>1) {
        return 1;
    }
    return x;
}

// Helpful averaging function.
float average(float *array, int N) {
    float sum=0;
    for(int k=0;k<N;k++) {
        sum+=array[k];
    }
    return (sum/N);
}