// NSDL library support functions

#include "mbed.h"
#include "nsdl_support.h"
#include "Logger.h"

extern Logger *m_logger;
#ifdef NETWORK_MUTEX
extern Mutex *network_mutex;
#endif

Endpoint nsp;
UDPSocket server;
char _endpoint_name[PERSONALITY_NAME_LEN+1];
uint8_t _ep_type[] = { NSP_NODE_TYPE };
uint8_t _lifetime_ptr[] = { NSP_NODE_LIFETIME };

#include "MBEDEndpoint.h"
extern MBEDEndpoint *endpoint;

void *nsdl_alloc(uint16_t size) {
    void *chunk = NULL;
    if (size > 0) chunk = malloc(size);
    if (chunk != NULL && size > 0) memset(chunk,0,size);
    return chunk;
}

void nsdl_free(void* ptr_to_free) {
    if (ptr_to_free != NULL) free(ptr_to_free);
}

void nsdl_create_static_resource(sn_nsdl_resource_info_s *resource_structure, uint16_t pt_len, uint8_t *pt, uint16_t rpp_len, uint8_t *rpp_ptr, uint16_t if_len, uint8_t *if_ptr, uint8_t is_observable, uint8_t is_registered, uint8_t *rsc, uint16_t rsc_len) {
    if (resource_structure != NULL) {
        resource_structure->access = SN_GRS_GET_ALLOWED;
        resource_structure->mode = SN_GRS_STATIC;
        resource_structure->pathlen = pt_len;
        resource_structure->path = pt;
        resource_structure->resource_parameters_ptr->resource_type_len = rpp_len;
        resource_structure->resource_parameters_ptr->resource_type_ptr = rpp_ptr;
        resource_structure->resource_parameters_ptr->interface_description_ptr = if_ptr;
        resource_structure->resource_parameters_ptr->interface_description_len = if_len;
        resource_structure->resource_parameters_ptr->observable = is_observable;
        resource_structure->resource_parameters_ptr->registered = is_registered;
        resource_structure->resource = rsc;
        resource_structure->resourcelen = rsc_len;
        sn_nsdl_create_resource(resource_structure);
    }
}

void nsdl_create_dynamic_resource(sn_nsdl_resource_info_s *resource_structure, uint16_t pt_len, uint8_t *pt, uint16_t rpp_len, uint8_t *rpp_ptr, uint16_t if_len, uint8_t *if_ptr, uint8_t is_observable, uint8_t is_registered, sn_grs_dyn_res_callback_t callback_ptr, int access_right) {
    if (resource_structure != NULL) {
        resource_structure->access = (sn_grs_resource_acl_e)access_right;
        resource_structure->resource = 0;
        resource_structure->resourcelen = 0;
        resource_structure->sn_grs_dyn_res_callback = callback_ptr;
        resource_structure->mode = SN_GRS_DYNAMIC;
        resource_structure->pathlen = pt_len;
        resource_structure->path = pt;
        resource_structure->resource_parameters_ptr->resource_type_len = rpp_len;
        resource_structure->resource_parameters_ptr->resource_type_ptr = rpp_ptr;
        resource_structure->resource_parameters_ptr->interface_description_ptr = if_ptr;
        resource_structure->resource_parameters_ptr->interface_description_len = if_len;
        resource_structure->resource_parameters_ptr->observable = is_observable;
        resource_structure->resource_parameters_ptr->registered = is_registered;
        sn_nsdl_create_resource(resource_structure);
    }
}

sn_nsdl_ep_parameters_s* nsdl_init_register_endpoint(sn_nsdl_ep_parameters_s *endpoint_structure, uint8_t* name, uint8_t* typename_ptr, uint8_t *lifetime_ptr) {
    if (endpoint_structure == NULL) {   
        endpoint_structure = (sn_nsdl_ep_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_ep_parameters_s));
    }
    
    if (endpoint_structure != NULL) {
        memset(endpoint_structure, 0, sizeof(sn_nsdl_ep_parameters_s));
        endpoint_structure->endpoint_name_ptr = name;
        endpoint_structure->endpoint_name_len = strlen((char*)name);
        endpoint_structure->type_ptr = typename_ptr;
        endpoint_structure->type_len =  strlen((char*)typename_ptr);
        endpoint_structure->lifetime_ptr = lifetime_ptr;
        endpoint_structure->lifetime_len =  strlen((char*)lifetime_ptr);
    }
    
    return endpoint_structure;
}

void nsdl_clean_register_endpoint(sn_nsdl_ep_parameters_s **endpoint_structure) {
    if (endpoint_structure != NULL) {
        if (*endpoint_structure != NULL) {
            nsdl_free(*endpoint_structure);
            *endpoint_structure = NULL;
        }
    }
}

static uint8_t tx_cb(sn_nsdl_capab_e protocol, uint8_t *data_ptr, uint16_t data_len, sn_nsdl_addr_s *address_ptr) {
    //if (m_logger != NULL) m_logger->log("NSP: sending %d bytes...",data_len);
    int sent = server.sendTo(nsp, (char*)data_ptr, data_len);

    //if (m_logger != NULL) m_logger->log("NSP: send done. sent %d bytes...",sent);
    if (sent != data_len) {
       if (m_logger != NULL) m_logger->log("NSP: send failed!! Attempted: %d Sent: %d", data_len, sent);
    }
    else {
       if (m_logger != NULL) m_logger->blinkTransportTxLED();
    }

    return 1;
}

static uint8_t rx_cb(sn_coap_hdr_s *coap_packet_ptr, sn_nsdl_addr_s *address_ptr) {
    // Rx callback process it...
    if (m_logger != NULL) m_logger->blinkTransportRxLED();
    return 0;
}

static void register_endpoint(bool init) {
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)_endpoint_name, _ep_type, _lifetime_ptr);
    sn_nsdl_update_registration(endpoint_ptr);
    nsdl_clean_register_endpoint(&endpoint_ptr);
}

static void registration_update_thread(void const *args) {
    int count = 0;
    int registration_time = RD_UPDATE_PERIOD;      // about 2.5 minutes assuming 4 iterations/second
    
    // first we want to wait a bit... let the endpoint crank up...
    Thread::wait(RD_UPDATE_PERIOD);   // wait about 150 seconds, then go ahead and start the re-registration interval...  
                
    // now loop this thread forever sleeping and re-registering at the right iteration.
    while(true) {
        Thread::wait(MAIN_LOOP_SLEEP);
        ++count;
        if (count%registration_time == 0) {
            // re-registration time!
            count = 0;
            if (m_logger != NULL) m_logger->log("NSP: (re)registering...");
            register_endpoint(false);
            if (m_logger != NULL) m_logger->log("NSP: (re)registering complete.");
        }
        else {
            if (count%RD_UPDATE_PERIOD == 0) {
                if (endpoint != NULL) endpoint->sendObservations();
            }
            if (m_logger != NULL) {
                 m_logger->checkForExit();
            }
        }
        if (m_logger != NULL) m_logger->blinkTransportRxLED();
    }
}

void nsdl_init() {    
    uint8_t nsp_addr[4];
    sn_nsdl_mem_s memory_cbs;
    
    /* Initialize libNsdl */
    memset(&memory_cbs,0,sizeof(memory_cbs));
    memset(nsp_addr,0,sizeof(nsp_addr));
    memory_cbs.sn_nsdl_alloc = &nsdl_alloc;
    memory_cbs.sn_nsdl_free = &nsdl_free;
    if(sn_nsdl_init(&tx_cb, &rx_cb, &memory_cbs) == -1) {
        if (m_logger != NULL) m_logger->log("NSP: libNsdl init failed");
    }
    else {
        if (m_logger != NULL) m_logger->log("NSP: libNsdl init successful");
    }

    /* Set nsp address for library */
    set_NSP_address(nsp_addr, NSP_PORT, SN_NSDL_ADDRESS_TYPE_IPV4);
}

// NSP event loop - spawn a re-registration thread AFTER we have initially registered and begun event processing...     
void nsdl_event_loop() {    
    sn_nsdl_addr_s received_packet_address; 
    Endpoint from;
    uint8_t nsp_received_address[4];
    char nsp_buffer[1024];

    memset(&received_packet_address, 0, sizeof(sn_nsdl_addr_s));
    memset(nsp_received_address, 0, sizeof(nsp_received_address));
    received_packet_address.addr_ptr = nsp_received_address;    
            
#ifdef ENABLE_THREADS
    // start the registration update thread.. it will wait a bit while the endpoint gins up...
    Thread registration_thread(registration_update_thread);
#endif
    
    // FOREVER: main loop for event processing  
    while(true) {        
        //if (m_logger != NULL) m_logger->log("NSP: waiting for data...");
        int n = server.receiveFrom(from,nsp_buffer,sizeof(nsp_buffer));

        //if (m_logger != NULL) m_logger->log("NSP: received %d bytes... processing...",n);
        if (n >= 0) sn_nsdl_process_coap((uint8_t*)nsp_buffer,n,&received_packet_address);
        
        //if (m_logger != NULL) m_logger->log("NSP: done processing %d bytes...",n);
        if (m_logger != NULL) m_logger->blinkTransportRxLED();

        // perform any extra event loop work
        if (endpoint != NULL) endpoint->extraEventLoopWork();
     }
}