#include "mbed.h"
#include "EthernetInterface.h"
#include "nsdl_support.h"
#include "dbg.h"
#include "IAP.h"

// Include resources
#include "LEDbar.h"
#include "IPSO_illuminance.h"
#include "gas_sensor.h"
#include "IPSO_presence.h"

Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "10.0.0.199"
#define MASK    "255.255.255.0"
#define GW      "10.0.0.1"

extern "C" void mbed_mac_address(char *mac)
{
    static char buf[64] = {0};
    IAP iap;
    int32_t *block = iap.read_serial();
    uint32_t serial_number[5] = {0};
    
    memset(buf, 0, sizeof(buf));
    serial_number[0] = *(block);
    serial_number[1] = *(block+1);
    // we only want bottom 16 bits of word1 (MAC bits 32-47)
    // and bit 9 forced to 1, bit 8 forced to 0
    // Locally administered MAC, reduced conflicts
    // http://en.wikipedia.org/wiki/MAC_address
    //serial_number[0] |= 0x00000200;
    //serial_number[0] &= 0x0000FEFF;
    memcpy(mac, (uint8_t*) &serial_number[0], 6);
    mac[0] |= 0x02;
    mac[0] &= 0xFE;
    mac[5] |= 0x02;
    mac[5] &= 0xFE;
     
    // snprintf(buf, 16, "%4X%08X", serial_number[0], serial_number[1]);
}  

 
// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
//static const char* NSP_ADDRESS = "23.102.162.118"; // coen296.cloudapp.net
static const char* NSP_ADDRESS = "191.239.5.150"; // barista2.cloudapp.net
static const int NSP_PORT = 5683;
char endpoint_name[24] = "mbedDEMO-";
uint8_t ep_type[] = {"DEMO"};
uint8_t lifetime_ptr[] = {"60"};

// ****************************************************************************
// Ethernet initialization

EthernetInterface eth;
static void ethernet_init()
{
    
    /* Initialize network */
#ifdef DHCP
    NSDL_DEBUG("DHCP in use\r\n");
    eth.init();
    NSDL_DEBUG("eth.init\r\n");
#else
    eth.init(IP, MASK, GW);
    NSDL_DEBUG("eth.init\r\n");
#endif
    if(eth.connect(30000) == 0)
        pc.printf("Connect OK\n\r");

    NSDL_DEBUG("IP Address:%s ", eth.getIPAddress());
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

char macbytes[6] = {0};
char MAC[20];

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    mbed_mac_address(macbytes);
    
    sprintf(MAC, "%02X%02X%02X%02X%02X%02X", macbytes[0], macbytes[1], macbytes[2], macbytes[3], macbytes[4], macbytes[5]);
    pc.printf("MAC: %s\r\n", MAC);
    
    strncat(endpoint_name, MAC, 12);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);

}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/0")-1, (uint8_t*) "3/0/0", 0, 0,  (uint8_t*) "mbedDEMO", sizeof("mbedDEMO")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/1")-1, (uint8_t*) "3/0/1", 0, 0,  (uint8_t*) "DEMO", sizeof("DEMO")-1);

    // Dynamic resources
    create_LEDbar_resource(resource_ptr);
    create_IPSO_illuminance_resource(resource_ptr);
    create_gas_sensor_resource(resource_ptr);
    create_IPSO_presence_resource(resource_ptr);

        /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        pc.printf("NSP registering failed\r\n");
    else
        pc.printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

int main()
{

    NSDL_DEBUG("Hello mDS Demo Endpoint Application\n");
    
    // Initialize Ethernet interface first
    ethernet_init();
    
    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
