/* Copyright C2014 ARM, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #ifndef _SALESFORCE_INTERFACE_H_
 #define _SALESFORCE_INTERFACE_H_
 
 // ErrorHandler
 #include "ErrorHandler.h"
 
 // SSL-based HTTP support
 #include "HTTPClient.h"
 
 // JSON parsing support
 #include "MbedJSONValue.h"
 
 // verbose debugging
 #if ENABLE_DEBUG_LOGGING
    #define DEBUG(...) { this->logger()->logConsole(__VA_ARGS__); }
 #else
    #define DEBUG(...)
 #endif 
 
 // convenience macros
 #define  RESET_BUFFER(x)       memset(x,0,MAX_BUFFER_LENGTH+1)
 #define  ALLOC_BUFFER(x)       char x[MAX_BUFFER_LENGTH+1];RESET_BUFFER(x)
 
 // HTTP Verbs
 typedef enum {
     GET,
     PUT,
     POST,
     DELETE,
     NUM_VERBS
 } HttpVerb;
 
 // Supported input data types for PUT and POST (Defined by HTTPClient-SSL/data support...)
 typedef enum {
     JSON,              // ContentType: application/json
     PLAIN_TEXT,        // ContentType: plain/text
     FORM_MAPPED,       // ContentType: application/x-www-form-urlencoded
     NUM_TYPES
 } InputDataTypes;
 
 // OAUTH structure
 typedef struct {
     bool   valid;
     string id;
     string issued_at;
     string token_type;
     string instance_url;
     string signature;
     string access_token;
 } OauthToken;
        
 // This class provides an interface into the REST-based Salesforce.com APIs
 class SalesforceInterface {
    private:
        ErrorHandler    *m_logger;
        HTTPClient      *m_http;
        char            *m_username;
        char            *m_password;
        char            *m_client_id;
        char            *m_client_secret;
        bool             m_have_creds;
        OauthToken       m_oauth_token;
        HTTPResult       m_http_status;
        int              m_http_response_code;
        char             m_http_redirection_url[MAX_BUFFER_LENGTH+1];
        
    public:
        // construction/destruction
        SalesforceInterface(ErrorHandler *logger,HTTPClient *http); 
        virtual ~SalesforceInterface();
        
        // set Salesforce.com credentials
        void setCredentials(char *username,char *password,char *client_id,char *client_secret);
        
        // get our ID structure from Salesforce
        char *getSalesforceID(char *output_buffer,int output_buffer_length);
                        
        // invoke REST calls into Salesforce.com
        char *invoke(const char *url,char *output_buffer,int output_buffer_len);                                                                               // defaults to GET
        char *invoke(const char *url,const char *input_data,const int input_data_len,char *output_buffer,int output_buffer_len);                                           // defaults to POST with JSON input data type                                                                              // defaults to GET
        char *invoke(const char *url,const InputDataTypes input_type,const char *input_data,const int input_data_len,char *output_buffer,int output_buffer_len);                  // defaults to POST with variable input data type
        char *invoke(const char *url,const InputDataTypes input_type,const char *input_data,const int input_data_len,char *output_buffer,int output_buffer_len,const HttpVerb verb);    // full fidelity method
        
    private:
        // get our OAUTH Token
        void checkAndGetOauthToken();
        char *getOauthToken(char *output_buffer,int output_buffer_length);
        
        // convenience accessors
        ErrorHandler *logger();
        HTTPClient *http();
        OauthToken *oauth();
        HTTPResult httpStatus();
        int httpResponseCode();
        
        // internal checkers
        bool haveCreds();
        void resetOauthToken();
        void fillOauthToken(char *token);
        bool validOauthToken();
 };
 
 #endif // _SALESFORCE_INTERFACE_H_