#include "mbed.h"
#include "EthernetInterface.h"
#include "C12832_lcd.h"
#include "nsdl_support.h"
#include "dbg.h"
// Include various resources
#include "temperature.h"
#include "light.h"
#include "gps.h"
#include "relay.h"
#include "rgb.h"

static C12832_LCD lcd;
Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP 1

// Define this to enable the RGB LED resource
//#define USE_RGBLED 1

/* Manual IP configurations, if DHCP not defined */
#define IP      "10.45.0.206"
#define MASK    "255.255.255.0"
#define GW      "10.45.0.1"

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
static const char* NSP_ADDRESS = "137.135.13.28"; /* demo NSP, web interface at http://red-hat-summit.cloudapp.net*/ 
static const int NSP_PORT = 5683;
char endpoint_name[20] = "mbed-ethernet-";
char mbed_uid[33]; // for creating unique name for the board
uint8_t ep_type[] = {"mbed_lpc1768_appboard"};
uint8_t lifetime_ptr[] = {"60"};
static const char* FIRMWARE_VER = "18"; // Committed revision number
char* mac;
char* ipAddr;
char* gateway;
char* nmask;

// ****************************************************************************
// Ethernet initialization

EthernetInterface eth;

static void ethernet_init()
{

    /* Initialize network */
#ifdef DHCP
    NSDL_DEBUG("DHCP in use");
    eth.init();
#else
    eth.init(IP, MASK, GW);
#endif
    if(eth.connect(30000) == 0)
        NSDL_DEBUG("Ethernet up");

    mac = eth.getMACAddress();
    ipAddr = eth.getIPAddress();
    gateway = eth.getGateway();
    nmask = eth.getNetworkMask();
    NSDL_DEBUG("Network: mac=%s, ip=%s, gateway=%s, mask=%s", mac, ipAddr, gateway, nmask);

    mbed_interface_uid(mbed_uid);
    mbed_uid[32] = '\0';
    NSDL_DEBUG("Full interface uid=%s", mbed_uid);
    strncat(endpoint_name, mbed_uid + 27, 20 - strlen(endpoint_name));

    lcd.locate(0,11);
    lcd.printf("IP:%s", eth.getIPAddress());

}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);

    lcd.locate(0,22);
    lcd.printf("EP:%s\n", endpoint_name);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/0")-1, (uint8_t*) "3/0/0", 0, 0,  (uint8_t*) "ARM", sizeof("ARM")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/1")-1, (uint8_t*) "3/0/1", 0, 0,  (uint8_t*) "LPC1768 App Board", sizeof("LPC1768 App Board")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/2")-1, (uint8_t*) "3/0/2", 0, 0,  (uint8_t*) mbed_uid, sizeof(mbed_uid)-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/3")-1, (uint8_t*) "3/0/3", 0, 0,  (uint8_t*) FIRMWARE_VER, strlen(FIRMWARE_VER));
   #ifdef DHCP
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/0")-1, (uint8_t*) "4/0/0", 0, 0,  (uint8_t*) "Ethernet DHCP", sizeof("Ethernet DHCP")-1);
   #else
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/0")-1, (uint8_t*) "4/0/0", 0, 0,  (uint8_t*) "Ethernet Static", sizeof("Ethernet Static")-1);  
   #endif
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/1")-1, (uint8_t*) "4/0/1", 0, 0,  (uint8_t*) "Ethernet (Static, DHCP)", sizeof("Ethernet (Static, DHCP)")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/4")-1, (uint8_t*) "4/0/4", 0, 0,  (uint8_t*) ipAddr, strlen(ipAddr));
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/5")-1, (uint8_t*) "4/0/5", 0, 0,  (uint8_t*) gateway, strlen(gateway));
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/16")-1, (uint8_t*) "3/0/16", 0, 0,  (uint8_t*) "UDP", sizeof("UDP")-1);

    // Dynamic resources
    create_temperature_resource(resource_ptr);
    create_light_resource(resource_ptr);
    create_gps_resource(resource_ptr);
    create_relay_resource(resource_ptr);

#ifdef USE_RGBLED
    NSDL_DEBUG("Enabling RGB LED due to USE_RGBLED=%d\n", USE_RGBLED);
    create_rgb_resource(resource_ptr);
#else
    NSDL_DEBUG("Skipped RGB LED resource, change USE_RGBLED to 1 in main.cpp to test");
#endif

    /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        pc.printf("NSP registering failed\r\n");
    else
        pc.printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

int main()
{
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("NanoService LWM2M r%s", FIRMWARE_VER);
    NSDL_DEBUG("NanoService LWM2M Demo for LPC1768 App Board\n");
    
    // Initialize Ethernet interface first
    ethernet_init();
    
    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
