/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
 
#include "MPL3115A2.h"

#define REG_WHO_AM_I        0x0C        // return 0xC4 by default
#define REG_STATUS          0x00
#define REG_CTRL_REG_1      0x26
#define REG_CTRL_REG_3      0x28
#define REG_CTRL_REG_4      0x29
#define REG_CTRL_REG_5      0x2A
#define REG_PRESSURE_MSB    0x01        // 3 byte pressure data
#define REG_ALTIMETER_MSB   0x01        // 3 byte altimeter data
#define REG_TEMP_MSB        0x04        // 2 byte temperature data
#define REG_PT_DATA_CFG     0x13
#define REG_P_TGT_MSB       0x16
#define REG_P_WND_MSB       0x19

// Status flag for data ready.
#define PTDR_STATUS       0x03        // Pressure Altitude and Temperature ready
#define PDR_STATUS        0x02        // Pressure and Altitude data ready
#define TDR_STATUS        0x01        // Temperature data ready

MPL3115A2::MPL3115A2(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr) {
    unsigned char data[2] = {REG_CTRL_REG_1, 0xB8};
    m_i2c.frequency(375000);
    writeRegs(data, 2);
    data[0] = REG_PT_DATA_CFG;
    data[1] = 0x07;
    writeRegs(data, 2);
    data[0] = REG_CTRL_REG_1;
    data[1] = 0xB9;
    writeRegs(data, 2);
}

uint8_t MPL3115A2::isDataAvailable( void)
{
    unsigned char status;
    readRegs( REG_STATUS, &status, 1);
    return ((status>>1) & 0x07);
    
}

uint8_t MPL3115A2::getAltimeterRaw( unsigned char *dt)
{
    if ( isDataAvailable() & PDR_STATUS)
    {
        readRegs( REG_ALTIMETER_MSB, &dt[0], 2);
        return 1;    
    }
    else
        return 0;
}

uint8_t MPL3115A2::getTemperatureRaw( unsigned char *dt)
{
    if ( isDataAvailable() & TDR_STATUS)
    {
        readRegs( REG_TEMP_MSB, &dt[0], 1);
        return 1;
    }
    else
        return 0;
}

void MPL3115A2::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void MPL3115A2::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}
