
/* Trinket/Gemma compatible Raw IR decoder sketch
This sketch/program uses an Adafruit Trinket or Gemma
ATTiny85 based mini microcontroller and a PNA4602 to
decode IR received. This can be used to make a IR receiver
(by looking for a particular code) or transmitter
(by pulsing an IR LED at ~38KHz for the durations detected

Based on Adafruit tutorial http://learn.adafruit.com/ir-sensor/using-an-ir-sensor

and ATTiny program by TinyPCRemote Nathan Chantrell http://nathan.chantrell.net
under Creative Commons Attribution-ShareAlike 3.0 Unported (CC BY-SA 3.0) license

Modified to work on MBed by David Bottrill May 2015
This has been tested on a KL25z using a Grove IR sensor with a number of different
Infrared Remote controls.

*/
#include "mbed.h"

#define MAXPULSE    10000                               // the maximum pulse we'll listen for - 10 milliseconds 
#define NUMPULSES    200                                // max IR pulse pairs to sample
#define RESOLUTION     1                                // time between IR measurements ~1uS

// we will store up to 200 pulse pairs (this is -a lot-)
uint16_t pulses[200][2];                                // pair is high and low pulse
uint16_t currentpulse = 0;                              // index for pulses we're storing
uint32_t irCode = 0;
Serial pc(USBTX, USBRX);                                // tx, rx
DigitalIn IRpin (p8);                                   // Listen to IR receiver on Trinket/Gemma pin D2

int listenForIR(void);                                  // IR receive code
void printcode(void);                                   // Print IR code in HEX

int main(void)
{
    pc.printf("Ready to decode IR!\n");

    while (1) {

        int numpulse=listenForIR();                     // Wait for an IR Code

        // Process the pulses to get a single number representing code
        for (int i = 0; i < 32; i++) {
            irCode=irCode<<1;
            if((pulses[i][0] * RESOLUTION)>0&&(pulses[i][0] * RESOLUTION)<500) {
                irCode|=0;
            } else {
                irCode|=1;
            }
        }

        printcode();                                    // Print IR code 
        wait_ms(100);

    }  //end of main loop
}


int listenForIR(void){                                  // IR receive code
    currentpulse = 0;
    while (1) {
        unsigned int highpulse, lowpulse;               // temporary storage timing
        highpulse = lowpulse = 0;                       // start out with no pulse length

        while (IRpin==1) {                              // got a high pulse
            highpulse++;
            wait_us(RESOLUTION);
            if (((highpulse >= MAXPULSE) && (currentpulse != 0))|| currentpulse == NUMPULSES) {
                return currentpulse;
            }
        }
        pulses[currentpulse][0] = highpulse;

        while (IRpin==0) {                              // got a low pulse
            lowpulse++;
            wait_us(RESOLUTION);
            if (((lowpulse >= MAXPULSE) && (currentpulse != 0))|| currentpulse == NUMPULSES) {
                return currentpulse;
            }
        }
        pulses[currentpulse][1] = lowpulse;
        currentpulse++;
    }
}

void printcode(void)
{
    uint16_t half;
    half=irCode>>16;                                    // Get first 16 bits of code
                           // Print upper 16 bits in hex
    pc.printf("%x\n", irCode);                // print lower 16 bits in hex
}

