/*
* Los Putacos
* Copyright (C) 2017, All rights reserved.
* ________________________________________
*
* Created by: Gustavo Campana, Michael Schmidt, Miguel Lopez
*       Date: 02-Dec-2017
*    Version: V0.1
*/
//-----------------------------------------------------------------

//-----------------------------------------------------------------
// "THE BEER-WARE LICENSE" (Revision 42):
// The "Los Putacos" team wrote this file. As long as you retain this notice you
// can do whatever you want with this stuff. If we meet some day, and you think
// this stuff is worth it, you can buy me a beer in return Poul-Henning Kamp
//-----------------------------------------------------------------

//-----------------------------------------------------------------
// Includes
#include "mbed.h"
#include "event.h"
#include "MPU9250.h"
#include "TCPSocket.h"
#include "configuration.h"
#include "ESP8266Interface.h"

//-----------------------------------------------------------------

//-----------------------------------------------------------------
// General Declarations
Timer TimeStamp;                // Timer µS time-stamp
Serial PC(USBTX, USBRX);        // Create an Serial PC Object - USBTX, USBRX

// IMU-SPI Declaration
SPI spi(IMU_MOSI, IMU_MISO,IMU_SCK);        // Create an SPI Object for MPU9250 - MOSI, MISO, SCK
mpu9250_spi imu(spi, IMU_CS);               // Create an MPU9250 Object - SPI Object, CS

// WiFi & TCP Socket Declaration
ESP8266Interface WiFi(ESP_TX, ESP_RX);      // ESP8266 WiFi Interface
TCPSocket Socket_TCP;                       // TCP Socket
nsapi_error_t response;                     // TCP Socket Response

// EventQueue Declarations
EventQueue queue(32 * EVENTS_EVENT_SIZE);   // Event Setup

// Ticker Declarations
Ticker Ticker_IMU;
Ticker Ticker_ReadBattery;
Ticker Ticker_ReceiveCommand;

// Thread Declarations
Thread Thread_IMU(osPriorityRealtime);
Thread Thread_ReadBattery(osPriorityRealtime);
Thread Thread_ReceiveCommand(osPriorityRealtime);

// Global Variables
uint16_t LineCount = 0;
uint32_t Line_Size = 0;
uint32_t Data_Size = 0;

// Read & Writepointer Declarations
uint8_t Current_Position = 0;
volatile uint16_t readPointer = 0;
volatile uint16_t readPointer_MIC = 0;
volatile uint16_t writePointer = 0;
volatile uint16_t writePointer_MIC = 0;

// Flag Declarations
uint8_t ReadIMUDone_Flag = 0;
uint8_t CheckCommandDone_Flag = 0;

// Battery Voltage & Power Consumption Declaration
float Battery_Status = 0;
float Power_Consumption = 0;

// Time Buffer Declaration
char Time_Buffer[2];
uint16_t Time_Data = 0;
uint8_t Requested_Time = 0;

// Data Storage Declaration
char Data_Buffer[1024];
const char DeviceNr[6] = "DEV01";       // Device specific ID
int16_t Data_Storage[BufferSize];               // BufferSize defined in "event.h"
uint16_t Data_Storage_MIC[BufferSize_MIC];      // BufferSize_MIC defined in "event.h"

//-----------------------------------------------------------------

//-----------------------------------------------------------------
void Setup()
{
    LED_Blue1 = 1;      // Turn OFF all LEDs
    LED_Blue2 = 1;
    LED_Red1 = 1;
    LED_Red2 = 1;

    PC.baud(230400);            // Initialize PC Serial Connection
    PC.printf("\r\n------------- Booting! -------------\r\n");
    PC.printf("CPU SystemCoreClock is %d Hz", SystemCoreClock);     // Show SystemCoreClock Speed

    PC.printf("\nConnecting to %s with %s...\n",  WiFi_SSID, WiFi_Pass);
    int ret = WiFi.connect(WiFi_SSID, WiFi_Pass, NSAPI_SECURITY_WPA_WPA2);      // Connect WiFi

    while (ret != 0) {          // WiFi Connection Not Successful
        PC.printf("\nConnecting to %s with %s...\n",  WiFi_SSID, WiFi_Pass);
        ret = WiFi.connect(WiFi_SSID, WiFi_Pass, NSAPI_SECURITY_WPA_WPA2);
        wait(0.5);
    }

    printf("Connection Done! - IP: %s\r\n", WiFi.get_ip_address());     // Show IP Address
    Socket_TCP.open(&WiFi);             // Open TCP_Socket on WiFi
    Socket_TCP.set_blocking(false);     // Set Non-Blocking Mode

    // Initialize IMU-SPI Connection
    if(imu.init(1, BITS_DLPF_CFG_188HZ))        // Initialize the MPU9250
        PC.printf("\nCouldn't initialize MPU9250 via SPI!");
    PC.printf("\nWHOAMI = 0x%2x", imu.whoami());                        // Output I2C Address to check SPI (correct: 104 - 0x68)
    PC.printf("\nAcc_Scale = %u\n", imu.set_acc_scale(BITS_FS_16G));    // Set Full Range for Acc.
    imu.calib_acc();        // Calibrate Acceleration Sensor

    ReadBattery();          // Read Battery Level
    TimeStamp.reset();      // Reset Timer TimeStamp
    Thread_ReceiveCommand.start(callback(&queue, &EventQueue::dispatch_forever));       // Start Command-Receiving Thread
    Ticker_ReceiveCommand.attach(queue.event(&ReceiveCommand), 0.500);                  // Attach 500ms Ticker to "ReceiveCommand"

    Thread_ReadBattery.start(callback(&queue, &EventQueue::dispatch_forever));          // Start Battery-Reading Thread
    Ticker_ReadBattery.attach(queue.event(&ReadBattery), 30);                           // Attach 30s Ticker to "ReadBattery"
    
    Buzzer = 0;             // Disable Buzzer
    Buzzer.period(0.001);   // 1kHz Buzzer Frequency
    
    LED_Blue1 = 0;          // Turn ON LED to display 'Booting Done!'
    PC.printf("\r\n------------- Ready! ---------------\r\n");
}

//-----------------------------------------------------------------

//-----------------------------------------------------------------
void Reset()            // Reset All Variables
{
    Line_Size = 0;
    Data_Size = 0;
    LineCount = 0;
    Time_Data = 0;
    Requested_Time = 0;
    Current_Position = 0;

    readPointer = 0;
    readPointer_MIC = 0;
    writePointer = 0;
    writePointer_MIC = 0;

    ReadIMUDone_Flag = 0;
    CheckCommandDone_Flag = 0;

    memset(Data_Storage, 0, sizeof(Data_Storage));
    memset(Data_Storage_MIC, 0, sizeof(Data_Storage_MIC));
}

//-----------------------------------------------------------------

//-----------------------------------------------------------------
int main()
{
    Setup();    // Initial Setups

    while (true) {
        if ((Requested_Time != 0) && (CheckCommandDone_Flag == 1)) {        // Check Flag & Requested Time
            CheckCommandDone_Flag = 0;          // Reset Flag
            memset(Data_Buffer, 0, sizeof(Data_Buffer));                    // Clear Data_Buffer

            TimeStamp.start();                  // Start Timer TimeStamp
            Thread_IMU.start(callback(&queue, &EventQueue::dispatch_forever));      // Start "ReadIMU" Thread
            Ticker_IMU.attach_us(queue.event(&ReadIMU), 2000);                      // Attach 2ms Ticker to "ReadIMU"
        }

        while ((writePointer != readPointer) && (writePointer_MIC != readPointer_MIC) && (ReadIMUDone_Flag == 1)) {     // Check Flag & Pointer Position
            while ((Data_Size + Line_Size) < 1024) {        // Check Maximum Package Size
                if ((writePointer != readPointer) && (writePointer_MIC != readPointer_MIC)) {
                    Data_Size += sprintf(&Data_Buffer[Data_Size], "%d;%d;%d;%d;%d\n", Data_Storage[readPointer++], Data_Storage_MIC[readPointer_MIC++], Data_Storage[readPointer++], Data_Storage[readPointer++], Data_Storage[readPointer++]);
                    Line_Size = snprintf(NULL, 0, "%d;%d;%d;%d;%d\n", Data_Storage[readPointer], Data_Storage_MIC[readPointer_MIC], Data_Storage[readPointer], Data_Storage[readPointer], Data_Storage[readPointer]);
                } else
                    break;
            }

            response = Socket_TCP.send(Data_Buffer, Data_Size);        // Send Data via TCP_Socket
            while (response != Data_Size)                              // Sending Not Successful
                response = Socket_TCP.send(Data_Buffer, Data_Size);    // Resend Data via TCP_Socket
                
            Data_Size = 0;      // Reset Data_Size
            memset(Data_Buffer, 0, sizeof(Data_Buffer));    // Clear Data_Buffer
            wait(0.1);          // Wait 100ms between Packages for ESP8266 Module

            if ((readPointer == writePointer) && (writePointer_MIC == readPointer_MIC)) {       // Check if Readpointer reached Writepointer Position
                ReadIMUDone_Flag = 0;           // Reset Flag for Sending
                PC.printf("Sending Done! - Time Taken: %d ms\n\n", TimeStamp.read_ms());        // Show Time used for Data Sending
                TimeStamp.stop();           // Stop Timer TimeStamp
                TimeStamp.reset();          // Reset Timer TimeStamp

                Reset();        // Reset All Variables
                Thread_ReceiveCommand.start(callback(&queue, &EventQueue::dispatch_forever));       // Start "ReceiveCommand" Thread
                Ticker_ReceiveCommand.attach(queue.event(&ReceiveCommand), 0.500);                  // Attach 500ms Ticker to "ReceiveCommand"

                Thread_ReadBattery.start(callback(&queue, &EventQueue::dispatch_forever));       // Start "ReadBattery"
                Ticker_ReadBattery.attach(queue.event(&ReadBattery), 30);                        // Attach 30s Ticker to "ReadBattery"
            }
        }
    }
}

//-----------------------------------------------------------------

//-----------------------------------------------------------------