/*
Author: Jan Conradie
Date: 09-04-2019

The following software is written specifically for the Lochtron Local Oscillator using the LMX2954 chip.
The PCB in question is documented as LLO_20G_100u.

The following libraries are PCB specific: 
Attenuator library
AM2005_ATTEN.h
AM2005_ATTEN.c

LED, RF switches and power line  control library
DIGITAL_CTRL.h
DIGITAL_CTRL.c

The following libraries are meant to be portable to other projects:
Synthesiser library
LMX2594_PLL.h
LMX2594_PLL.c

EEPROM library
25LC1024_EEP.h
25LC1024_EEP.c

*/

//Include libraries
#include "mbed.h"
#include "main.h"
#include "math.h"
#include "stdlib.h"
#include "LMX2594_PLL.h"
//#include "AM2005_ATTEN.h"
//#include "DIGITAL_CTRL.h"


//#include "25LC1024_EEP.h"

//Initialise serial port.
Serial sbus(MBED_SBUS_TX,MBED_SBUS_RX);    //SBUS conneciton
Serial USB(USBTX , USBRX); // USB to serial connection from the PC to the Mbed
//Serial uart(MBED_REG_TX,MBED_SBUS_RX);  //Regular UART


//Initialise IOs:

//Digital Out
DigitalOut DOUBLER_EN(MBED_DD_EN);
DigitalOut OSC_EN(MBED_OSC_EN);
DigitalOut AMP1_EN(MBED_AMP1_EN);
DigitalOut AMP2_EN(MBED_AMP2_EN);

DigitalOut ATT_PIN_A(MBED_ATT_A);
DigitalOut ATT_PIN_B(MBED_ATT_B);
DigitalOut ATT_PIN_C(MBED_ATT_C);
DigitalOut ATT_PIN_D(MBED_ATT_D);
DigitalOut ATT_PIN_E(MBED_ATT_E);

DigitalOut LED_PCB(MBED_LED1);
DigitalOut LED_STATUS_GREEN(MBED_LED2);
DigitalOut LED_STATUS_RED(MBED_LED3);

DigitalOut SW1(MBED_SW1);
DigitalOut MSW1(MBED_MSW1);
DigitalOut MSW2(MBED_MSW2);

//LMX Debug IO (Digital Inputs)
DigitalIn RDIR_LMX(MBED_RDIR_SYNTH);
DigitalIn RCLK_LMX(MBED_RCLK_SYNTH);

//Analogue Inputs
AnalogIn ADC_TEMP_LMX(MBED_TEMP_LMX);
AnalogIn ADC_PEAK_DET(MBED_PEAK_DET_IN);

//DAC Output
AnalogOut DAC_OSC(MBED_DAC);

//SPI PIN initialise for LMX2594
DigitalOut CS_LMX(MBED_CSB_LMX);
DigitalOut CLK_LMX(MBED_CLK_LMX);
DigitalOut MOSI_LMX(MBED_MOSI_LMX);
DigitalIn MUX_LMX(MBED_MUX_SYNTH); //MISO



int main()
{
  
  // Variables 

  //unsigned short PLL_REG[113] = {0}; //Create main SPI register 
  
  //Setup serial ports
  USB.baud(115200);
  sbus.baud(115200);
  //uart.baud(115200);

  // Wait for board startup
  wait(1);
  LMX2594_Load_Defaults(LM2594_PLL_reg);   //Load LMX2594 default register values
  
  Startup_Splash();
  
  //Device ready to use
  while(1)
  {
        //Set_FrequencyA(7000.10); //Euraka when possible
    
  }
  
}

void Startup_Splash(void)
{
  if(DEBUG)
  {
    USB.printf("LLO_20G_100u Starting up\r\n");
    USB.printf("Software version 1.0\r\n");
  }  
  
}

void LMX2594_Spi_Write(char address, unsigned short data)
{
  CS_LMX.write(0); 
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  MOSI_LMX.write(0); //Set R/W bit to 0 for write
  CLK_LMX.write(0);
  
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  CLK_LMX.write(1); //Complete first bit transmission
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  
  char address_length = 7; // Adapt data type to be the correct bit length
  unsigned short data_length = 16;
  
  for(int l = address_length-1; l >= 0; l--) //Start at the second MSB
  { 
      CLK_LMX.write(0);
      MOSI_LMX = (address>> l) & 0x01; //Shift bits through from MSB
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
      CLK_LMX.write(1);
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  }   
  
  for(int l = data_length-1; l >= 0; l--)
  { 
      CLK_LMX.write(0);
      MOSI_LMX = (data>> l) & 0x01; //Shift bits through from MSB 
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
      CLK_LMX.write(1);
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  }   
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  CS_LMX.write(1); //End transaction
}


int LMX2594_Spi_Read(char address)
{
  CS_LMX.write(0);  
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  
  MOSI_LMX.write(1); //Set R/W bit to 1 for read.
  CLK_LMX.write(0);

  wait_us(SPI_CLK_SPEED); //delay is determined by slave device 
  CLK_LMX.write(1); //Complete first bit transmission
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  char address_length = 7; // Adapt data type to be the correct bit length
  unsigned short data_length = 16;
  int spi_data_value = 0; // Adapt data type to be the correct bit length

  
  for(int l = address_length-1; l >= 0; l--) //Start at the second MSB
  { 
      CLK_LMX.write(0);
      MOSI_LMX = (address>> l) & 0x01; //Shift bits through from MSB
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
      CLK_LMX.write(1);
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  }   
  
  for(int l = data_length-1; l >= 0; l--)
  { 
      CLK_LMX.write(0);
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
      spi_data_value = spi_data_value + (MUX_LMX.read() << l);
      CLK_LMX.write(1);
      wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  }  
   
  CS_LMX.write(1); //End transaction
  wait_us(SPI_CLK_SPEED); //delay is determined by slave device
  
  return spi_data_value;
}

void LMX2594_Read_All_Registers(void)
{
  //Ensure that LMX2594 MUX pin is in read mode
  char spi_address = 0x00;
  unsigned short spi_data = 0x2518;
  LMX2594_Spi_Write(spi_address, spi_data);  
  
  char read_address = 0x70; //112
  //Read all registers
  for(int c=112; c>=0; c--)
  {
    USB.printf("Register[%d] = %x\r\n",c,LMX2594_Spi_Read(read_address));
    read_address--;
  }
}