//TDC1000.h
//More plain-english functions for changing settings and stuff
//Created on 7 December 2015, by Eric Lindholm

#include "mbed.h"
#include "TI_Registers.h"
        
#ifndef TDC1000_H
#define TDC1000_H

///The SettingChoice variable indicates which setting should be changed.
//It allows the program to look up the how the setting in question is used in a table.
//This only applies to on/off settings.
//Unfortunately, some of these settings are true=on and some true=off.
enum SettingChoice{VCOM_SEL, MEAS_MODE, DAMPING, CH_SWP, EXT_CHSEL, CH_SEL, TEMP_MODE,
        TEMP_RTD_SEL, TEMP_CLK_DIV, BLANKING, RECEIVE_MODE, TRIG_EDGE_POLARITY,
        PGA_CTRL, LNA_CTRL, LNA_FB, FORCE_SHORT_TOF, ECHO_TIMEOUT, CLOCKIN_DIV};
const int settingAddress[18] = {0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x03, 0x03,
         0x03, 0x03, 0x04, 0x04, 0x05, 0x05, 0x05, 0x08, 0x08, 0x09};
const int settingLookUp[18] =  {0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x40, 0x20, 
         0x10, 0x08, 0x40, 0x20, 0x10, 0x08, 0x04, 0x40, 0x04, 0x04};

/** TDC1000.h
 *   Chip function manifest for the TDC1000
 *   More documentation to come.
 *    Unfortunately, there are a lot of pins that need to be determined.
 *   This file assumes that a separate stopwatch chip is used, and 
 *    that the MCU only performs the register access functions plus a 
 *    few more.  If an invalid number is entered for a 'set' function, 
 *    the setting will be set to the default.
 *   Pin Number on TDC1000           Function
 *   11                              Channel Select.  Use any digital out pin for this.
 *   12                              Error pin.  Requires an INTERRUPT pin in.
 *   15                              Enable pin.  Use any digital out pin for this.
 *   17                              Reset pin.  Use any digital out pin for this.
 *   18                              SPI serial clock (SCLK) pin.  Tie to an SPI interface.
 *   19                              Chip select pin.  Tied to SPI signal, but can be any digital out pin.
 *   20                              SPI MOSI serial pin.  Tie to an SPI interface.
 *   21                              SPI MISO serial pin.  Tie to an SPI interface.
 */
class TDC1000
{
//Put all functions here, as well as register object.
public:
   //!Create TDC1000 instance
    TDC1000(PinName ChSel = p22, PinName ERR = p23, PinName EN = p16, PinName RESET = p10, 
            PinName SCLK = p7, PinName CS = p8, PinName MOSI = p5, PinName MISO = p6);
    
    //Unknown if this should be public or private.  Probably private.
    TI_Registers tdc1000_registers;
    
    //Error handler function
   /** readError Function
    *   The read error function reads the error register and returns that value.
    */
    int readError();
    
    /**  errorHandler function
    *   The error handler function is not really meant to be called by the user.
    *   It details the problem the TDC1000 has claimed, but does very little about it.
    */
    void errorHandler();
    
   /** readToggle function
    *   The read a toggle setting function reads a toggle setting for the TDC1000.
    *   There are ...a few toggle settings for the TDC1000.
    *   NOTICE: a value of true does not always equal enabled...
    *   
    *    Use the name that is in the TI documentation, with capital letters, of that setting.
    *    Or, you can refer to this big chart.
    *   VCOM_SEL                |   Common-mode voltage reference control (default internal)
    *   MEAS_MODE               |   AFE measurement type (default time-of-flight)
    *   DAMPING                 |   Transmit burst damping (default disabled)
    *   CH_SWP                  |   Turns on or off automatic channel swapping in mode 2 (default disabled)
    *   EXT_CHSEL               |   Allows for external channel selection (default disabled)
    *   CH_SEL                  |   Selects the active transmit/receive pair (default channel 1)
    *   TEMP_MODE               |   Select which temperature measurement channels are used (default Reference, RTD1 and RTD2)
    *   TEMP_RTD_SEL            |   Change which type of RTD is currently connected (default PT1000)
    *   TEMP_CLK_DIV            |   Modify the clock divider for the temperature mode (default divide by 8)
    *   BLANKING                |   Adds power blanking in the standard TOF measurements (default disabled)
    *   RECEIVE_MODE            |   Changes receive echo mode (default: single echo)
    *   TRIG_EDGE_POLARITY      |   Change what the trigger edge polarity should be (default rising edge)
    *   PGA_CTRL                |   Turns on or off the Programmable Gain Amplifier (default active)
    *   LNA_CTRL                |   Turns on or off the Low-Noise Amplifier (default active)
    *   LNA_FB                  |   Modifies the LNA feedback mode (default capacitive feedback)
    *   FORCE_SHORT_TOF         |   Forces the chip to use a short TOF mode even with a large timing register (default disabled)
    *   ECHO_TIMEOUT            |   Turns on or off the timeout function, which means the chip will stay in receiving mode until 
    *                           |     it receives all of the necessary echoes (default enabled)
    *   CLOCKIN_DIV             |   The divisor for generating the standard waiting period (default 1)
    *
    *   @param choice Choice indicates which setting you want to read.
    */
    bool readToggle(SettingChoice);
    
   /** setToggle function
    *   This toggle function is very similar to the readToggle function, except that 
    *    instead of simply reading the setting value, it changes it.  If the value was on,
    *    this function turns it off, and if it was off, this function turns it on.
    *    Just like how toggles should work.
    *
    *   @param choice Choice indicates which setting you want to toggle.
    */
    void setToggle(SettingChoice);
    
    //Now, more specific functions
    /** MODE_SELECT is the mode the chip is in.
    *   The mode of the chip selects which transducers transmit and which 
    *    transducer receives the signal.
    *   Mode 0 uses each transducer as both sender and receiver, and maps the 
    *    transmit1 channel with receive2, and transmit2 with receive1. In this 
    *    mode, Channel selection is what tells the chip which transducer channel
    *    to use: ChSel = 0, transmit1; ChSel = 1, transmit2.  In this mode,
    *    the chip is used primarily for fluid level measurement.
    *   Mode 1 also uses each transducer as both sender and receiver.  This 
    *    mode varies with mode 0 by mapping the transmit1 channel with the 
    *    receive1 channel, and transmit2 with receive2.  Otherwise it works 
    *    almost exactly like mode 0.
    *   Mode 2 is different in that it is for fluid flow measurement.  The 
    *    channels are mapped in the same way as mode 1 (TX1-RX1 & TX2-RX2), but
    *    when a signal is transmitted on one channel, the chip will be listening
    *    for the echo on the other channel. It can be thought of as sending 
    *    signals back-and-forth between the two transducers.  Mode 2 allows for
    *    automatic channel swapping: as soon as the signal goes one way, the
    *    chip can automatically send it through the second channel in the other
    *    direction.  For automatic swapping, EXT_CHSEL needs to be off.
    *   There is no mode 3 ... yet.
     */
     int readMODE_SELECT();
     void setMODE_SELECT(int);
    //! TX_FREQ_DIV is the transmit frequency divisor.
    //! Transmit Frequency = (Frequency_in)/(2^(TX_FREQ_DIV+1))
    int readTX_FREQ_DIV();
    void setTX_FREQ_DIV(int);
    //! NUM_TX is the number of transmit pulses.
    int readNUM_TX();
    void setNUM_TX(int);
    //! NUM_AVG is the number of cycles to create averaging.
    int readNUM_AVG();
    void setNUM_AVG(int);
    //! NUM_RX is the number of expected receive pulses.
    int readNUM_RX();
    void setNUM_RX(int);
    //! ECHO_QUAL_THLD is the voltage that qualifies a receive pulse.
    //! the value is a number between 0 and 7 that controls the actual threshold value, which is in negative millivolts.
    int readECHO_QUAL_THLD();
    void setECHO_QUAL_THLD(int);
    //! TX_PH_SHIFT_POS is the point in the transmit burst that the phase shift occurs.
    int readTX_PH_SHIFT_POS();
    void setTX_PH_SHIFT_POS(int);
    //! PGA_GAIN is the gain of the Programmable Gain Amplifier.
    //! The actual gain is the gain number * 3dB.
    int readPGA_GAIN();
    void setPGA_GAIN(int);
    //! TIMING_REG creates the period between transmit and receive.
    //The timing register is a bit different than the rest of these.
    int readTIMING_REG();
    void setTIMING_REG(int);
    //! SHORT_TOF_BLANK_PERIOD is the blanking period for a short TOF measurement.
    int readSHORT_TOF_BLANK_PERIOD();
    void setSHORT_TOF_BLANK_PERIOD(int);
    //! TOF_TIMEOUT_CTRL is the echo listening window timeout.
    int readTOF_TIMEOUT_CTRL();
    void setTOF_TIMEOUT_CTRL(int);
    //! AUTOZERO_PERIOD is the receiver auto-zero period.
    int readAUTOZERO_PERIOD();
    void setAUTOZERO_PERIOD(int);
    
    //Other functions that don't play with the registers.
    //! Enable the chip
    bool read_EN();
    void set_EN(bool);
    
    //! Reset the chip, also resetting all settings.
    void resetTDC1000();
    
    //! utilize external channel selection 
    bool readChannelSelect();
    void toggleChannelSelect();
    
    //! Functions that read setting values
    int readClockFrequencyIn();
    int readTransmitFrequencyDivision();
    double readEchoQualificationThreshold();
    int readTransmitPhaseShiftPosition();
    double readPeriod0();
    double readPeriod1();
    int readTimingRegister();
    int readNumberofCycles();
    double readShortTOFBlankingPeriod();
    double readAutoZeroPeriod();
    
    //! Clock frequency is not set within the chip.
    //! @param newfreq is the frequency of the system clock in Hz.
    void setClockFrequencyIn(int);
    
//stuff that shouldn't be touched goes here.
private:
    //A few variables, like what mode is currently active and the actual digital out stuff
    DigitalOut en;
    DigitalOut chsel;
    DigitalOut reset;
    InterruptIn error;
    int currentMode;
    int fclkin, txfreqdiv, txphshiftpos, timingreg, numavg;
    double echoqualthld, t0, t1, shorttofblankperiod, autozeroperiod;
};
#endif