//TDC1000.cpp
//Class and functions for clearer settings of values
//Created on 8 December 2015 by Eric Lindholm

#include "mbed.h"
#include "TDC1000.h"
#include "TI_Registers.h"
#include <cstring>

/** TDC1000 class
*   There are a total of 8 pins to the TDC1000 if you have a separate stopwatch chip.
*   If you use the MCU as a stopwatch chip, which is not recommended, the number 
*    increases to 11.
*   The constructor for this class has 8 pin inputs and initializes a whole bunch
*    of things, including the default values for a few variables.
*   Pin Number on TDC1000           Function
*   11                              Channel Select.  Use any digital out pin for this.
*   12                              Error pin.  Requires an INTERRUPT pin in.
*   15                              Enable pin.  Use any digital out pin for this.
*   17                              Reset pin.  Use any digital out pin for this.
*   18                              SPI serial clock (SCLK) pin.  Tie to an SPI interface.
*   19                              Chip select pin.  Tied to SPI signal, but can be any digital out pin.
*   20                              SPI MOSI serial pin.  Tie to an SPI interface.
*   21                              SPI MISO serial pin.  Tie to an SPI interface.
*/
TDC1000::TDC1000(PinName ChSel, PinName ERR, PinName EN, PinName RESET, PinName SCLK, PinName CS, PinName MOSI, PinName MISO):
        chsel(ChSel), error(ERR), en(EN), reset(RESET), tdc1000_registers(MOSI, MISO, SCLK, CS) {
    error.fall(this,&TDC1000::errorHandler);
    set_EN(true);
    chsel = 0;
    reset = 0;
    currentMode = 0;
    setClockFrequencyIn(16000000);
    txfreqdiv = 8;
    txphshiftpos = 31;
    timingreg = 0;
    numavg = 1;
    echoqualthld = -0.125;
    //t0 = (2^CLOCKIN_DIV)/fclkin
    t0 = 1 / ((double)fclkin);
    //t1 = (2^(TX_FREQ_DIV+1))/fclkin
    t1 = 8 / ((double)fclkin);
    shorttofblankperiod = 64 * t0;
    autozeroperiod = 64 * t0;
}

//readError function
int TDC1000::readError() {
    //The address of the error register is 0x07.
    int errnum = tdc1000_registers.registerRead8(0x07);
    //Next, we clear the error register.
    tdc1000_registers.registerWrite(0x07, 0x03);
    return errnum;
}

//determine error
void TDC1000::errorHandler() {
    int errnum = readError();
    char errorDesc[100];
    if(errnum & 0x04) 
        strcpy(errorDesc, "\nThe signal was too weak, and timeout has occurred.");
    if(errnum & 0x02) 
        strcpy(errorDesc, "\nNo signal was received at all and timeout has occurred.");
    if(errnum & 0x01) 
        strcpy(errorDesc, "\nThe signal was too large and overwhelmed the comparators.");
    else 
        strcpy(errorDesc, "\nUnknown error occurred.");
}

/** readToggle function
*   The read a toggle setting function reads a toggle setting for the TDC1000.
*   There are ...a few toggle settings for the TDC1000.
*   NOTICE: a value of true does not always equal enabled...
*   @param choice Choice indicates which setting you want to toggle.
*   
*    Use the name that is in the TI documentation, with capital letters, of that setting.
*    Or, you can refer to this big chart.
*   VCOM_SEL                |   Common-mode voltage reference control (default internal)
*   MEAS_MODE               |   AFE measurement type (default time-of-flight)
*   DAMPING                 |   Transmit burst damping (default disabled)
*   CH_SWP                  |   Turns on or off automatic channel swapping in mode 2 (default disabled)
*   EXT_CHSEL               |   Allows for external channel selection (default disabled)
*   CH_SEL                  |   Selects the active transmit/receive pair (default channel 1)
*   TEMP_MODE               |   Select which temperature measurement channels are used (default Reference, RTD1 and RTD2)
*   TEMP_RTD_SEL            |   Change which type of RTD is currently connected (default PT1000)
*   TEMP_CLK_DIV            |   Modify the clock divider for the temperature mode (default divide by 8)
*   BLANKING                |   Adds power blanking in the standard TOF measurements (default disabled)
*   RECEIVE_MODE            |   Changes receive echo mode (default: single echo)
*   TRIG_EDGE_POLARITY      |   Change what the trigger edge polarity should be (default rising edge)
*   PGA_CTRL                |   Turns on or off the Programmable Gain Amplifier (default active)
*   LNA_CTRL                |   Turns on or off the Low-Noise Amplifier (default active)
*   LNA_FB                  |   Modifies the LNA feedback mode (default capacitive feedback)
*   FORCE_SHORT_TOF         |   Forces the chip to use a short TOF mode even with a large timing register (default disabled)
*   ECHO_TIMEOUT            |   Turns on or off the timeout function, which means the chip will stay in receiving mode until 
*                           |     it receives all of the necessary echoes (default enabled)
*   CLOCKIN_DIV             |   The divisor for generating the standard waiting period (default 1)
*/
bool TDC1000::readToggle(SettingChoice choice) {
    int registerIn = tdc1000_registers.registerRead8(settingAddress[choice]);
    bool stateOfSetting = (bool)(registerIn & settingLookUp[choice]);
    return stateOfSetting;
}

/** setToggle function
*   Used for toggling a toggle setting for the TDC1000 on or off.
*/
void TDC1000::setToggle(SettingChoice choice) {
    int registerIn = tdc1000_registers.registerRead8(settingAddress[choice]);
    int newRegisterSetting = (registerIn ^ settingLookUp[choice]); 
    
    tdc1000_registers.registerWrite(settingAddress[choice], newRegisterSetting);
}

//SELECT_MODE
int TDC1000::readMODE_SELECT() {
    //The mode of the TDC1000 is the 1 and 0 bits of the config2 register.
    int registerInt = tdc1000_registers.registerRead8(0x02);
    currentMode = (registerInt & 0x03);
    return currentMode;
}
void TDC1000::setMODE_SELECT(int newMode) {
    int registerInt = tdc1000_registers.registerRead8(0x02);
    int registerPut;
    if((newMode < 3) && (newMode >=0)) {
        registerPut = (registerInt & 0xFC) + newMode;
        currentMode = newMode;
    }
    else {
        registerPut = (registerInt & 0xFC);
        currentMode = 0;
    }
    tdc1000_registers.registerWrite(0x02,registerPut);
}

//TX_FREQ_DIV
int TDC1000::readTX_FREQ_DIV() {
    //transmit frequency = fclkin / (2 ^ (txfreqdiv + 1))
    //the TX_FREQ_DIV is the 7, 6, and 5 bits of the config0 register.
    int registerInt = tdc1000_registers.registerRead8(0x00);
    txfreqdiv = (registerInt & 0xE0)>>4;
    return txfreqdiv;
}
void TDC1000::setTX_FREQ_DIV(int divisor) {
    //read the config0 register to determine current settings
    int registerInt = tdc1000_registers.registerRead8(0x00);
    int registerPut;
    if((divisor < 8) && (divisor >= 0)) {
        registerPut = (divisor<<5) + (registerInt & 0x1F);
        txfreqdiv = 2<<divisor;
    }
    else {
        registerPut = (registerInt & 0x1F) + 0x40;
        txfreqdiv = 8;
    }
    tdc1000_registers.registerWrite(0x00, registerPut);
}

//NUMTX
int TDC1000::readNUM_TX() {
    //the NUM_TX is the 4 thru 0 bits of the config0 register.
    int registerInt = tdc1000_registers.registerRead8(0x00);
    int numtx = (registerInt & 0x1F);
    return numtx;
}
void TDC1000::setNUM_TX(int numberOfPulses) {
    //the NUM_TX is the 4 thru 0 bits of the config0 register.
    int registerInt = tdc1000_registers.registerRead8(0x00);
    int registerPut;
    if((numberOfPulses < 32) && (numberOfPulses >= 0))
        registerPut = (registerInt & 0xE0) + (numberOfPulses);
    else
        registerPut = (registerInt & 0xE0) + 5;
    tdc1000_registers.registerWrite(0x00,registerPut);
}

//NUM_AVG
int TDC1000::readNUM_AVG() {
    //the NUM_AVG is the 5 thru 3 bits of the config1 register.
    int registerInt = tdc1000_registers.registerRead8(0x01);
    numavg = (registerInt & 0x38)>>2;
    return numavg;
}
void TDC1000::setNUM_AVG(int numberToAverage) {
    //the NUM_AVG is the 5 thru 3 bits of the config1 register.
    int registerInt = tdc1000_registers.registerRead8(0x01);
    int registerPut;
    if((numberToAverage < 8) && (numberToAverage >= 0))
        registerPut = (registerInt & 0x07) + 0x40 + (numberToAverage)<<3;
    else
        registerPut = (registerInt & 0x07) + 0x40 ;
    tdc1000_registers.registerWrite(0x01,registerPut);
    numavg = numberToAverage;
}

//NUMRX
int TDC1000::readNUM_RX() {
    //the NUM_RX is the 2 thru 0 bits of the config1 register.
    int registerInt = tdc1000_registers.registerRead8(0x01);
    int numrx = (registerInt & 0x1F);
    return numrx;
}
void TDC1000::setNUM_RX(int numberOfPulses) {
    //the NUM_RX is the 2 thru 0 bits of the config1 register.
    int registerInt = tdc1000_registers.registerRead8(0x01);
    int registerPut;
    if((numberOfPulses < 8) && (numberOfPulses >= 0))
        registerPut = (registerInt & 0x07) + 0x40 + (numberOfPulses);
    else
        registerPut = (registerInt & 0x07) + 0x40;
    tdc1000_registers.registerWrite(0x01,registerPut);
}

double echoThresholdArray[] = {-0.035, -0.050, -0.075, -0.125, -0.220, -0.410, -0.775, -1.500};

//ECHO_QUAL_THLD
int TDC1000::readECHO_QUAL_THLD() {
    //the ECHO_QUAL_THLD is the 2 thru 0 bits of the config3 register.
    int registerInt = tdc1000_registers.registerRead8(0x03);
    int echoqual = (registerInt & 0x07);
    
    echoqualthld = echoThresholdArray[echoqual];
    return echoqual;
}
void TDC1000::setECHO_QUAL_THLD(int tablevalue) {
    //the ECHO_QUAL_THLD is the 2 thru 0 bits of the config3 register.
    int registerInt = tdc1000_registers.registerRead8(0x03);
    int registerPut;
    if((tablevalue < 8) && (tablevalue >= 0))
        registerPut = (registerInt & 0x78) + (tablevalue);
    else
        registerPut = (registerInt & 0x78) + 3;
    tdc1000_registers.registerWrite(0x03,registerPut);
    echoqualthld = echoThresholdArray[tablevalue];
}

//TX_PH_SHIFT_POS
int TDC1000::readTX_PH_SHIFT_POS() {
    //the TX_PH_SHIFT_POS is the 4 thru 0 bits of the config4 register.
    int registerInt = tdc1000_registers.registerRead8(0x04);
    txphshiftpos = (registerInt & 0x1F);
    return txphshiftpos;
}
void TDC1000::setTX_PH_SHIFT_POS(int newphaseshiftposition) {
    //the TX_PH_SHIFT_POS is the 4 thru 0 bits of the config4 register.
    int registerInt = tdc1000_registers.registerRead8(0x04);
    int registerPut;
    if((newphaseshiftposition < 32) && (newphaseshiftposition >= 0)) {
        registerPut = (registerInt & 0xE0) + (newphaseshiftposition);
        txphshiftpos = newphaseshiftposition;
    }
    else
        registerPut = (registerInt & 0xE0) + 31;
    tdc1000_registers.registerWrite(0x04,registerPut);
}

//PGA_GAIN
int TDC1000::readPGA_GAIN() {
    //the PGA_GAIN is the 7 thru 5 bits of the TOF1 register.
    int registerInt = tdc1000_registers.registerRead8(0x05);
    int pgagain = (registerInt & 0xE0)>>5;
    return pgagain;
}
void TDC1000::setPGA_GAIN(int gain) {
    //the PGA_GAIN is the 7 thru 5 bits of the TOF1 register.
    int registerInt = tdc1000_registers.registerRead8(0x05);
    int registerPut;
    if((gain < 8) && (gain >= 0))
        registerPut = (registerInt & 0x1F) + (gain)<<5;
    else
        registerPut = (registerInt & 0x1F) + 0;
    tdc1000_registers.registerWrite(0x05,registerPut);
}

//TIMING_REG
int TDC1000::readTIMING_REG() {
    //The TIMING_REG is the 2 last bits of the TOF1 register AND all of the TOF0 register.
    int tof1bits = tdc1000_registers.registerRead8(0x05);
    int tof0bits = tdc1000_registers.registerRead8(0x06);
    timingreg = ((tof1bits & 0x03)<<8) + (tof0bits);
    return timingreg;
}
void TDC1000::setTIMING_REG(int timereg) {
    //The TIMING_REG is the 2 last bits of the TOF1 register and all of the TOF0 register.
    int tof1bits = tdc1000_registers.registerRead8(0x05);
    int tof0bits;
    if((timereg > 1024) && (timereg >= 0)) {
        tof1bits = (tof1bits & 0xFC) + timereg>>8;
        tof0bits = (timereg & 0xFF);
        timingreg = timereg;
    }
    else {
        tof1bits = (tof1bits & 0xFC);
        tof0bits = 0;
        timingreg = 0;
    }
    tdc1000_registers.registerWrite(0x05,tof1bits);
    tdc1000_registers.registerWrite(0x06,tof0bits);
}

//SHORT_TOF_BLANK_PERIOD
int TDC1000::readSHORT_TOF_BLANK_PERIOD() {
    //the SHORT_TOF_BLANK_PERIOD is the 5 thru 3 bits of the timeout register.
    int registerInt = tdc1000_registers.registerRead8(0x08);
    int shortblank = (registerInt & 0x38)>>3;
    shorttofblankperiod = (double)(4<<(shortblank + 1)) * t0;
    return shortblank;
}
void TDC1000::setSHORT_TOF_BLANK_PERIOD(int shortblank) {
    //the SHORT_TOF_BLANK_PERIOD is the 5 thru 3 bits of the timeout register.
    int registerInt = tdc1000_registers.registerRead8(0x08);
    int registerPut;
    if((shortblank < 8) && (shortblank >= 0))
        registerPut = (registerInt & 0x47) + (shortblank)<<3;
    else
        registerPut = (registerInt & 0x47) + 24;
    tdc1000_registers.registerWrite(0x08,registerPut);
    shorttofblankperiod = (double)(4<<(shortblank + 1)) * t0;
}

//TOF_TIMEOUT_CTRL
int TDC1000::readTOF_TIMEOUT_CTRL() {
    //the TOF_TIMEOUT_CTRL is the 1 thru 0 bits of the timeout register.
    int registerInt = tdc1000_registers.registerRead8(0x08);
    int toftimeout = (registerInt & 0x03);
    return toftimeout;
}
void TDC1000::setTOF_TIMEOUT_CTRL(int toftimeout) {
    //the TOF_TIMEOUT_CTRL is the 1 thru 0 bits of the timeout register.
    int registerInt = tdc1000_registers.registerRead8(0x08);
    int registerPut;
    if((toftimeout < 4) && (toftimeout >= 0))
        registerPut = (registerInt & 0x7C) + (toftimeout);
    else
        registerPut = (registerInt & 0x7C) + 1;
    tdc1000_registers.registerWrite(0x08,registerPut);
}

//AUTOZERO_PERIOD
int TDC1000::readAUTOZERO_PERIOD() {
    //the AUTOZERO_PERIOD is the 1 thru 0 bits of the clock rate register.
    int registerInt = tdc1000_registers.registerRead8(0x09);
    int autozero = (registerInt & 0x03);
    autozeroperiod = (double)(64<<autozero) * t0;
    return autozero;
}
void TDC1000::setAUTOZERO_PERIOD(int autozero) {
    //the AUTOZERO_PERIOD is the 1 thru 0 bits of the clock rate register.
    int registerInt = tdc1000_registers.registerRead8(0x09);
    int registerPut;
    if((autozero < 4) && (autozero >= 0))
        registerPut = (registerInt & 0x04) + (autozero);
    else
        registerPut = (registerInt & 0x04) + 0;
    tdc1000_registers.registerWrite(0x09,registerPut);
    autozeroperiod = (double)(64<<autozero) * t0;
}

//Next are functions that mess with non-register pins.
//The Enable pin
bool TDC1000::read_EN() {
    return en;
}
void TDC1000::set_EN(bool onOff) {
    if(onOff)
        en = 1;
    else
        en = 0;
}

//The reset pin
void TDC1000::resetTDC1000() {
    //Toggle reset pin
    reset = 0;
    wait_us(10);
    reset = 1;
    wait_us(10);
    reset = 0;
    
    //Reset variable values
    currentMode = 0;
    txfreqdiv = 8;
    txphshiftpos = 31;
    timingreg = 0;
    numavg = 1;
    echoqualthld = -0.125;
    //t0 = (2^CLOCKIN_DIV)/fclkin
    t0 = 1 / ((double)fclkin);
    //t1 = (2^(TX_FREQ_DIV+1))/fclkin
    t1 = (double)txfreqdiv / ((double)fclkin);
    shorttofblankperiod = 64 * t0;
    autozeroperiod = 64 * t0;
}

//The channel select pin
bool TDC1000::readChannelSelect() {
    return chsel;
}
void TDC1000::toggleChannelSelect() {
    if(chsel)
        chsel = 0;
    else
        chsel = 1;
}

//Functions that read setting values
int TDC1000::readClockFrequencyIn() {
    return fclkin;
}
int TDC1000::readTransmitFrequencyDivision() {
    return txfreqdiv;
}
double TDC1000::readEchoQualificationThreshold() {
    return echoqualthld;
}
int TDC1000::readTransmitPhaseShiftPosition() {
    return txphshiftpos;
}
double TDC1000::readPeriod0() {
    return t0;
}
double TDC1000::readPeriod1() {
    return t1;
}
int TDC1000::readTimingRegister() {
    return timingreg;
}
int TDC1000::readNumberofCycles() {
    return numavg;
}
double TDC1000::readShortTOFBlankingPeriod() {
    return shorttofblankperiod;
}
double TDC1000::readAutoZeroPeriod() {
    return autozeroperiod;
}

//The clock frequency is not really a chip setting, but it affects a lot of things.
void TDC1000::setClockFrequencyIn(int newfreq) {
    if(newfreq > 0) 
        fclkin = newfreq;
    else
        fclkin = 8000000;
}