//TI_Registers class
//By Eric Lindholm, created on 7 December 2015

#include "mbed.h"
#include "TI_Registers.h"

/** Ultrasonic_Registers class
 *   To access the registers of the ultrasonic driver chip: the TDC1000
 *   MOSI is the pin which is assigned as serial out from the MCU.
 *   MISO is the pin which is assigned as serial in to the MCU.
 *   SCLK is the pin which is assigned as serial clock.
 *   SPICS is the chip select pin for the SPI connection.
 */
TI_Registers::TI_Registers(PinName MOSI, PinName MISO, PinName SCLK, PinName SPICS) : device(MOSI, MISO, SCLK), cs(SPICS,1) {
    //Double-ensure that cs is normally HIGH
    cs = 1;
    
    //set the SPI device to use an 8-bit transfer.
    device.format(8);
    //set the SPI device to use 1 MHz clock.
    device.frequency(1000000);
}

/** registerRead8 function
 *   The registerRead functions individually read 1, 2, or 3 bytes of data.
 *   address is the address that is included in the TI documentation for the 
 *    register in question, and is 6 bits long.
 */
int TI_Registers::registerRead8(int address) {
    //Create variable for new contents
    int contents;
    
    //start the SPI connection
    cs = 0;
    
    //transfer address
    device.write(address);
    
    //read contents of register
    contents = device.write(0x00);
    
    //Finish SPI connection
    cs = 1;
    
    return contents;
}

/** registerRead16 function
 *   The registerRead functions individually read 1, 2, or 3 bytes of data.
 *   address is the address that is included in the TI documentation for the 
 *    register in question, and is 6 bits long.
 */
int TI_Registers::registerRead16(int address) {
    //Create variables for new contents
    int content1, content2;
    
    //start the SPI connection
    cs = 0;
    
    //transfer address
    device.write(address);
    
    //read contents of register
    content1 = device.write(0x00);
    content2 = device.write(0x00);
    
    //Finish SPI connection
    cs = 1;
    
    int contents = (content1<<8) + content2;
    return contents;
}

/** registerRead24 function
 *   The registerRead functions individually read 1, 2, or 3 bytes of data.
 *   address is the address that is included in the TI documentation for the 
 *    register in question, and is 6 bits long.
 */
int TI_Registers::registerRead24(int address) {
    //Create variables for new contents
    int content1, content2, content3;
    
    //start the SPI connection
    cs = 0;
    
    //transfer address
    device.write(address);
    
    //read contents of register
    content1 = device.write(0x00);
    content2 = device.write(0x00);
    content3 = device.write(0x00);
    
    //Finish SPI connection
    cs = 1;
    
    int contents = (content1<<16) + (content2<<8) + content3;
    return contents;
}

/** registerWrite function 
 *   The registerWrite function writes one byte of data to the register 
 *   address is the 8-bit address for the register in question, determined
 *    from the TI documentation for the chip: 6 bits.
 *   contents is the 8 bits of data that will be written to the register.
 */
void TI_Registers::registerWrite(int address, int contents) {
    //First, we append the write bit to the address.
    address += (1<<6);
    
    //Next, set the SPI connection active
    cs = 0;
    
    //Send the address to the chip
    device.write(address);
    
    //Write the data we need to the chip
    device.write(contents);
    
    //end SPI connection
    cs = 1;
}