//TDC7200.h
//More plain-english functions for changing settings and stuff.
//Created on 11 December 2015 by Eric Lindholm

#include "mbed.h"
#include "TI_Registers.h"

#ifndef TDC7200_H
#define TDC7200_H

/** The SettingChoice variable
 *   The setting choice variable allows the program to look up what register and 
 *    bit number a setting corresponds to.
 */
 //This only applies to on-off settings.
 enum SettingChoice7200{FORCE_CAL, PARITY_EN, TRIGG_EDGE, STOP_EDGE, START_EDGE,
        START_MEAS, CLOCK_CNTR_OVF_MASK, COARSE_CNTR_OVF_MASK, NEW_MEAS_MASK};
const int settingAddress7200[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x03, 0x03};
const int settingLookUp7200[] = {0x80, 0x40, 0x20, 0x10, 0x08, 0x01, 0x04, 0x02, 0x01};


/** TDC7200 class
 *   The TDC7200 class is a chip function manifest.  It controls and manipulates 
 *   settings, counters, interrupts, and masks.
 *   This file assumes that the appropriate pins are linked to the MCU.
 *   Also, any setting that is set to an invalid number will be set instead to the default.
 *   It should be noted that the TDC7200 gets its accuracy from the accuracy of its 
 *    clock driver.  So, a clock signal that varies by even 1% may give highly inaccurate times.
 *   Pinout Table from the TDC7200:
 *      1               |   Enable pin.  Turns chip on or off.
 *      8               |   Interrupt pin.  Signals when an event happens.
 *      9               |   MISO pin.  Goes to SPI driver.
 *      10              |   MOSI pin.  Goes to SPI driver.
 *      11              |   Chip select pin. Part of SPI system, but can go to any I/O pin.
 *      12              |   Serial clock pin. Part of SPI system, goes to SPI driver.
 */
class TDC7200
{
public:
    //! The constructor takes all the pinout data.
    TDC7200(PinName EN = p29, PinName INTRPT = p30, PinName MISO = p12, PinName MOSI = p11, PinName SCLK = p13, PinName SPICS = p14);
    
    //! Please don't go mucking about in the registers if you don't know what you're doing.
    TI_Registers tdc7200_registers;

    /** ReadToggleSetting
     *  The read toggle setting function reads a toggle setting.
     *  NOTICE: a value of 'true' does not always mean the setting is on.
     *  Chart for toggle settings:
     *  FORCE_CAL               |   Ensures chip performs calibration, even after an interrupted measurement (default off)
     *  PARITY_EN               |   Turns on the parity bits for the results (default off)
     *  TRIGG_EDGE              |   Changes between using the rising edge for the trigger or the falling edge (default rising edge)
     *  STOP_EDGE               |   Changes between using the rising edge for the stop signal or the falling edge (default rising edge)
     *  START_EDGE              |   Changes between using the rising edge for the start signal or the falling edge (default rising edge)
     *  START_MEAS              |   Tells the chip to begin a new measurement, also resets results registers (default off)
     *  CLOCK_CNTR_OVF_MASK     |   Turns on the interrupt for the clock counter overflow (default enabled)
     *  COARSE_CNTR_OVF_MASK    |   Turns on the interrupt for the coarse counter overflow (default enabled)
     *  NEW_MEAS_MASK           |   turns on the interrupt for when a new measurement is completed (default enabled)
     *
     *      @param choice indicates which setting should be changed.
     */
    bool readToggleSetting(SettingChoice7200); 
   /** setToggle function
    *   This toggle function is very similar to the readToggle function, except that 
    *    instead of simply reading the setting value, it changes it.  If the value was on,
    *    this function turns it off, and if it was off, this function turns it on.
    *    Just like how toggles should work.
    *
    *   @param choice Choice indicates which setting you want to toggle.
    */
    void setToggle(SettingChoice7200);
    
    //! More specific functions
    
    /** MODE_SELECT
     *  There are two modes for this chip: mode 1 and mode 2.
     *  Mode 1 is for when your expected time-of-flight is less than about 
     *   twenty clock cycles (~500 ns).
     *  Mode 2 is for when your expected time-of-flight is higher than that.
     *   But still probably less than 2000 clock cycles.
     *  The reason this isn't a toggle function is that this chip has the ability
     *   to have more than two modes.  But they aren't listed in the documentation.
     */
    int readMODE_SELECT();
    void setMODE_SELECT(int);
    
    /** CALIBRATION2_PERIODS
     *  This setting controls how many clock cycles the second calibration measurement
     *   will go through.
     *  ---
     *  setting value       |   Actual number of cycles
     *  0                   |   2 
     *  1                   |   10
     *  2                   |   20
     *  3                   |   40
     */
     int readCALIBRATION2_PERIODS();
     void setCALIBRATION2_PERIODS(int);
     
    /** AVG_CYCLES
     *  The setting for the number of measurement cycles the chip will average
     *   a reading through.  The number of cycles is equal to 2 ^ (setting value)
     */
    int readAVG_CYCLES();
    void setAVG_CYCLES(int);
    
    /** NUM_STOP
     *  This setting is the number of stop signals the TDC7200 expects to see.
     *   It can only count up to five stop signals, though.
     *  A value of zero is one stop, one means two stops, and so on.
     */
     int readNUM_STOP();
     void setNUM_STOP(int);
     
    //! Maintenance Functions
    
    /** The enable pin
     *  The enable pin is actually a little weird.  This pin has to be set to off when 
     *   the chip initially receives power, then turned on for the chip to work.
     */
    bool read_EN();
    void set_EN(bool);
    
    /// The interrupt function
    void interruptReceived();
    
    //! The clock frequency in
    void setClockFrequencyIn(int);
    int readClockFrequencyIn();
    
    //! Other miscellaneous values
    int readCalibrationPeriods();
    int readAverageCycles();
    int readNumberStops();
    /// for readTimeOfFlight, the parameter is WHICH time-of-flight you want.
    double readTimeOfFlight(int);
    double readNormLSB();
    double readCalibrationCount();

    //! Finally, the two functions I want to use the most.
    void forceMeasurementRead();
    void startMeasurement();
    
    /** Formulae
     *  calibrationCount = (calibration2 - calibration1) / (calibration2_periods -1)
     *  normLSB = 1 / ((clock frequency) * (calibrationCount))
     *  MODE 1:
     *   timeOfFlight[n] = (time[n] * normLSB)
     *  MODE 2:
     *   offset = (clock period) - (calibration1) * (normLSB)
     *   timeOfFlight[n] = (normLSB) * (time[1] - time[n+1]) + (clock count[n]) * (clock period)
     */
private:
    DigitalOut en;
    InterruptIn int_7200;
    int fclkin;
    int currentmode;
    bool parityBit;
    int calibrationPeriods, averageCycles, numberStops;
    double timeOfFlight[5], normLSB, calibrationCount;
};
#endif