//TDC7200.cpp
//Functions for the TDC7200 chip class
//Creaetd on 11 December 2015
//Eric Lindholm

#include "mbed.h"
#include "TI_Registers.h"
#include "TDC7200.h"

//The constructor start up the interrupt and the enable pin.
TDC7200::TDC7200(PinName EN, PinName INTRPT, PinName MISO, PinName MOSI, PinName SCLK, PinName SPICS) : 
        en(EN), int_7200(INTRPT), tdc7200_registers(MOSI, MISO, SCLK, SPICS) {

    //The enable pin has to see a positive edge after the device is powered up.
    set_EN(false);
    wait_us(5);
    set_EN(true);
    
    //Attach the interrupt handler
    int_7200.rise(this, &TDC7200::interruptReceived);
   
    //Set default values
    setClockFrequencyIn(8000000);
    currentmode = 0;        //Mode 1
    parityBit = false;
    calibrationPeriods = 10;
    averageCycles = 1;
    numberStops = 1;
    for(int q = 0; q < 6; q++)
        timeOfFlight[q] = 0;
    calibrationCount = 0;
    normLSB = 0;
}

//Read a toggle setting
bool TDC7200::readToggleSetting(SettingChoice7200 choice) {
    int registerInt = tdc7200_registers.registerRead8(settingAddress7200[choice]);
    bool stateOfSetting = (bool)(stateOfSetting & settingLookUp7200[choice]);
    return stateOfSetting;
}
//Toggle a setting
void TDC7200::setToggle(SettingChoice7200 choice) {
    int registerInt = tdc7200_registers.registerRead8(settingAddress7200[choice]);
    int newRegisterSetting = (registerInt ^ settingLookUp7200[choice]);
    tdc7200_registers.registerWrite(settingAddress7200[choice], newRegisterSetting);
}

//MODE_SELECT
int TDC7200::readMODE_SELECT() {
    //the mode is the 2 thru 1 bits of the CONFIG1 register
    int registerInt = tdc7200_registers.registerRead8(0x00);
    currentmode = (registerInt & 0x06)>>1;
    return currentmode;     //I feel like I should add one to this number.
}
void TDC7200::setMODE_SELECT(int newMode) {
    int registerInt = tdc7200_registers.registerRead8(0x00);
    int registerPut;
    if((newMode < 2) && ( newMode >= 0)) {
        registerPut = (registerInt & 0xF9) + (newMode<<1);
        currentmode = newMode;
    }
    else {
        registerPut = (registerInt & 0xF9);
        currentmode = 0;
    }
    tdc7200_registers.registerWrite(0x00, registerPut);
}

//CALIBRATION2_PERIODS
int TDC7200::readCALIBRATION2_PERIODS() {
    //the calibration2 periods is the 7 thru 6 bits of the CONFIG2 register
    int registerInt = tdc7200_registers.registerRead8(0x01);
    int calib2_periods = (registerInt & 0xC0)>>6;
    if(calib2_periods == 0)
        calibrationPeriods = 2;
    else if(calib2_periods == 1)
        calibrationPeriods = 10;
    else if(calib2_periods == 2)
        calibrationPeriods = 20;
    else if(calib2_periods == 3)
        calibrationPeriods = 40;
    else
        calibrationPeriods = 10;
    return calib2_periods;
}
void TDC7200::setCALIBRATION2_PERIODS(int newCalibPeriods) {
    int registerInt = tdc7200_registers.registerRead8(0x01);
    int registerPut;
    if((newCalibPeriods < 4) && ( newCalibPeriods >= 0)) {
        registerPut = (registerInt & 0x3F) + (newCalibPeriods<<6);
        if(newCalibPeriods == 0)
            calibrationPeriods = 2;
        else if(newCalibPeriods == 1)
            calibrationPeriods = 10;
        else if(newCalibPeriods == 2)
            calibrationPeriods = 20;
        else if(newCalibPeriods == 3);
            calibrationPeriods = 40;
    }
    else {
        registerPut = (registerInt & 0x3F);
        calibrationPeriods = 10;
    }
    tdc7200_registers.registerWrite(0x01, registerPut);
}

//AVG_CYCLES
int TDC7200::readAVG_CYCLES() {
    //the average cycles is the 5 thru 3 bits of the CONFIG2 register
    int registerInt = tdc7200_registers.registerRead8(0x01);
    int numcycles = (registerInt & 0x38)>>3;
    averageCycles = 1<<(numcycles);
    return numcycles;
}
void TDC7200::setAVG_CYCLES(int newCycles) {
    int registerInt = tdc7200_registers.registerRead8(0x01);
    int registerPut;
    if((newCycles < 8) && ( newCycles >= 0)) {
        registerPut = (registerInt & 0xC7) + (newCycles<<3);
        averageCycles = 1<<(newCycles);
    }
    else {
        registerPut = (registerInt & 0xC7);
        averageCycles = 1;
    }
    tdc7200_registers.registerWrite(0x01, registerPut);
}

//NUM_STOP
int TDC7200::readNUM_STOP() {
    //the number of stops is the 2 thru 0 bits of the CONFIG2 register
    int registerInt = tdc7200_registers.registerRead8(0x01);
    int numstops = (registerInt & 0x07);
    numberStops = numstops + 1;
    return numstops;
}
void TDC7200::setNUM_STOP(int newStops) {
    int registerInt = tdc7200_registers.registerRead8(0x01);
    int registerPut;
    if((newStops < 8) && ( newStops >= 0)) {
        registerPut = (registerInt & 0xF8) + (newStops);
        numberStops = newStops + 1;
    }
    else {
        registerPut = (registerInt & 0xF8) ;
        numberStops = 1;
    }
    tdc7200_registers.registerWrite(0x01, registerPut);
}

//Enable Pin
bool TDC7200::read_EN() {
    return en;
}
void TDC7200::set_EN(bool onOff) {
    if(onOff)
        en = 1;
    else
        en = 0;
}

//Called when the interrupt happens
void TDC7200::interruptReceived() {
    //First, check what interrupt event happened.
    int eventInt = tdc7200_registers.registerRead8(0x02);
    //bit 0: a new measurement has been completed.
    //bit 1: a coarse counter overflow happened.
    //bit 2: a clock counter overflow happened.
    //bit 3: a measurement was started (START pin triggered)
    //bit 4: a measurement has been completed.
    //Bits 5, 6, and 7 do not have anything.
    int clearBits = 0;
    if(eventInt & 0x01) {
        //New measurement completed
        forceMeasurementRead();
        clearBits += 1;
    }
    if(eventInt & 0x02) {
        //coarse counter overflow
        clearBits += 2;
    }
    if(eventInt & 0x04) {
        //clock counter overflow
        clearBits += 4;
    }
    if(eventInt & 0x08) {
        //measurement started
        clearBits += 8;
    }
    if(eventInt & 0x10) {
        //Measurement completed
        forceMeasurementRead();
        clearBits += 16;
    }
    //clear the interrupted flags
    tdc7200_registers.registerWrite(0x02, clearBits);
}

//Clock frequency
int TDC7200::readClockFrequencyIn() {
    return fclkin;
}
void TDC7200::setClockFrequencyIn(int newFreq) {
    if(newFreq > 0)
        fclkin = newFreq;
    else
        fclkin = 8000000;
}

//other
int TDC7200::readCalibrationPeriods() {
    return calibrationPeriods;
}
int TDC7200::readAverageCycles() {
    return averageCycles;
}
int TDC7200::readNumberStops() {
    return numberStops;
}
double TDC7200::readNormLSB() {
    return normLSB;
}
double TDC7200::readCalibrationCount() {
    return calibrationCount;
}
//The Time of flight data
double TDC7200::readTimeOfFlight(int index = 0) {
    if((index < 5) && (index >= 0))
        return timeOfFlight[index];
    else
        return timeOfFlight[0];
}

//Different kinds of functions
//Read all the measurement registers and do calculations
void TDC7200::forceMeasurementRead() {
    //Read in calibration values
    int calib1 = tdc7200_registers.registerRead24(0x1B);
    int calib2 = tdc7200_registers.registerRead24(0x1C);
    //Read in time1 through time6 as well as the clock counts.
    int timeregister[numberStops + 1];
    int clockcount[numberStops];
    int regaddress = 0x10;
    for(int j = 0; j < numberStops; j++) {
        timeregister[j] = tdc7200_registers.registerRead24(regaddress);
        clockcount[j] = tdc7200_registers.registerRead24(regaddress +  1);
        regaddress += 2;
    } 
    timeregister[numberStops] = tdc7200_registers.registerRead24(0x10 + 2 * numberStops);
    //If the parity bit is enabled...
    if(parityBit) {
        int timeparity[numberStops + 1];
        int clockparity[numberStops];
        bool paritybits[2 * numberStops + 3];
        //Go through each measurement and check the parity (even parity).
        //Then re-request every measurement that didn't match parity.
        int parityselection = 0x800000;
        int calib1parity = calib1 & 0x7FFFFF;
        paritybits[2 * numberStops + 1] = false;
        while(calib1parity) {
            paritybits[2 * numberStops + 1] = !paritybits[2 * numberStops + 1];
            calib1parity = calib1parity & (calib1parity - 1);
        }
        if(((calib1 & parityselection)>>23) != paritybits[2 * numberStops + 1]) 
            calib1 = tdc7200_registers.registerRead24(0x1B);
            
        int calib2parity = calib2 & 0x7FFFFF;
        paritybits[2 * numberStops + 2] = false;
        while(calib2parity) {
            paritybits[2 * numberStops + 2] = !paritybits[2 * numberStops + 2];
            calib2parity = calib2parity & (calib2parity - 1);
        }
        if(((calib1 & parityselection)>>23) != paritybits[2 * numberStops + 1]) 
            calib1 = tdc7200_registers.registerRead24(0x1B);
            
        timeparity[0] = timeregister[0] & 0x7FFFFF;
        paritybits[0] = false;
        while(timeparity[0]) {
            paritybits[0] = !paritybits[0];
            timeparity[0] = timeparity[0] & (timeparity[0] - 1);
        }
        if(((timeregister[0] & parityselection)>>23) != paritybits[0])
            timeregister[0] = tdc7200_registers.registerRead24(0x10);
            
        for(int k = 0; k < numberStops; k++) {
            timeparity[k+1] = timeregister[k+1] & 0x7FFFFF;
            paritybits[k+1] = false;
            while(timeparity[k+1]) {
                paritybits[k+1] = !paritybits[k+1];
                timeparity[k+1] = timeparity[k+1] & (timeparity[k+1] - 1);
            }
            if(((timeregister[k+1] & parityselection)>>23) != paritybits[k+1])
                timeregister[k+1] = tdc7200_registers.registerRead24(0x10 + 2 * (k + 1));
            
            clockparity[k] = clockcount[k] & 0x7FFFFF;
            paritybits[numberStops + k] = false;
            while(clockparity[k]) {
                paritybits[numberStops + k] = !paritybits[numberStops + k];
                clockparity[k] = clockparity[k] & (clockparity[k] - 1);
            }
            if(((clockcount[k] & parityselection)>>23) != paritybits[numberStops + k])
                clockcount[k] = tdc7200_registers.registerRead24(0x11 + 2 * k);
        }
    }       //This was all just for the parity bit calculations
    
    //Calculate the normLSB and CalibrationCount.
    calibrationCount = ((double)calib2 - (double)calib1) / ((double)calibrationPeriods - 1);
    normLSB = 1/ ((double)fclkin * calibrationCount);
    //Mode 1
    if(currentmode == 0) {
        for(int f = 0; f < numberStops; f++)
            timeOfFlight[f] = (double)(timeregister[f]) * normLSB;
    }
    //Mode 2
    else if(currentmode == 1) {
        for(int n = 0; n < numberStops; n++) 
            timeOfFlight[n] = ((double)timeregister[n] * normLSB) + (double)clockcount[n] / ((double)fclkin) - ((double)timeregister[n+1] * normLSB);
    }

}
//Begin a new measurement
void TDC7200::startMeasurement() {
    //read current state
    int registerInt = tdc7200_registers.registerRead8(0x00);
    //Add the 'start measurement' bit
    registerInt = (registerInt & 0xFE) + 1;
    //put that back into the chip
    tdc7200_registers.registerWrite(0x00, registerInt);
    
}