#include "mbed.h"
#include "edge_sensor.h"
#include "edge_time.h"
#include "edge_reset_mgr.h"
#include "PSE530.h"
#include "edge_pressure.h"
#include <ILI9341.h>
#include "edge_chart.h"
// #include "SO1602A.h"

// extern SO1602A *display ; /* OLED display on I2C */
extern ILI9341 *display ;

#define LOW_THR   0.2
#define HIGH_THR  0.3 
#define MIN_TEMP 12.0
#define MAX_TEMP 30.0

/**
 * SMC PSE530 pressure sensor
 * analog output 1.0V - 5.0V
 * 1.0V : 0
 * 5.0V : 1MPa
 * (at 0.6V : -0.1MPa)
 * Our sensor I/F converts 0-5V to 0-1
 * So we suppose V = Analog Float Value : Pressure
 * 0.2 =  0
 * 1.0 = 1MPa
 */
 
float temp2expected(float temp)
{
    const float coef_A = 0.089 ;
    const float coef_B = 0.831 ;
    float pressure ;

    pressure = temp * coef_A + coef_B ;
    return( pressure ) ;
}

edge_pressure::edge_pressure(PSE530 *pse, DigitalOut *en)
{
    _pse = pse ;
    _en = en ;
    _value = 0.0 ;
    _interval = 30 ;
    _thr_mode = 0 ;
    _thr_high = HIGH_THR ;
    _thr_low = LOW_THR ;
}

edge_pressure::~edge_pressure(void)
{
    if (_pse) {
        delete _pse ;
    }
    _value = 0.0 ;
}

float edge_pressure::get_value(void)
{
    float value = 0.0 ;
    value = _pse->getPressure() ;
    return( value ) ;
}

void edge_pressure::reset(void) 
{
    _value = 0.0 ;
    _sampled_time = 0 ;
}

void edge_pressure::prepare(void) 
{
}

int edge_pressure::sample(void) 
{
    int result = 0 ;
    *_en = 1 ; /* enable pressure sensor */
    wait_ms(30) ;
    _value = get_value() ;
    _sampled_time = edge_time ;
    *_en = 0 ; /* disable pressure sensor */
    wait_ms(10) ; /* to avoid power transition effect remaining */
    return( result ) ; /* this always success */
}

void edge_pressure::set_thr_high(int16_t thr_high)
{
    switch(_thr_mode) {
    case 0: /* absolute value */
        _thr_high = (float)thr_high/100.0 ;
        break ;
    case 1: /* persentage */
        _thr_high = (float)(thr_high/100.0) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        _thr_high = (float)thr_high/100.0 ;
        break ;
    }
// printf("thr_high = %.3f\n", _thr_high) ;
}

float edge_pressure::get_thr_high(float expected)
{
    float thr_high ;
    
    switch(_thr_mode) {
    case 0: /* absolute value */
        thr_high = expected + _thr_high ;
        break ;
    case 1: /* persentage */
        thr_high = expected * (1.0 + _thr_high) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        thr_high = expected + _thr_high ; /* use this as default */
        break ;
    }
    return (thr_high) ;
}

void edge_pressure::set_thr_low(int16_t thr_low)
{   
    switch(_thr_mode) {
    case 0: /* absolute value */
        _thr_low = (float)thr_low/100.0 ;
        break ;
    case 1: /* persentage */
        _thr_low = (float)(thr_low/100.0) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        _thr_low = (float)thr_low/100.0 ;
        break ;
    }
//printf("thr_low = %.3f\n", _thr_low) ;
}

float edge_pressure::get_thr_low(float expected)
{
    float thr_low ;
    
    switch(_thr_mode) {
    case 0: /* absolute value */
        thr_low = expected - _thr_low ;
        break ;
    case 1: /* persentage */
        thr_low = expected * (1.0 - _thr_low) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        thr_low = expected + _thr_low ; /* use this as default */
        break ;
    }
    return (thr_low) ;
}

int edge_pressure::deliver(void) 
{
    char str_buf[32] ;
    char timestr[16] ;
    int result ;

reset_watch_dog() ;
    print_time(_sampled_time) ;
    if (current_temp != 0) {
        sprintf(str_buf, "GAS: %.3f kgf/cm2 @ %.1fC", _value, *current_temp ) ;
    } else {
        sprintf(str_buf, "GAS: %.3f kgf/cm2", _value  ) ;
    }
    printf(" ") ;
    printf(str_buf) ;

    if (current_temp != 0) {
reset_watch_dog() ;
        _expected = temp2expected(*current_temp) ;
        _higher = get_thr_high(_expected) ;
        _lower = get_thr_low(_expected) ;
        printf(" (%.3f, %.3f) ", _higher, _lower) ;
    }

reset_watch_dog() ;
    printf(" %s\n", str_buf) ;
    time2seq(_sampled_time, timestr) ;
//    printf(str_buf) ;
//    printf("\n") ;
    sprintf(_str_buf,
       "{\"DEVICE\":\"PRESS\",\"PN\":\"PSE530\",\"VAL\":\"%.3f\",\"UNIT\":\"kgf/cm2\",\"T\":\"%s\",\"E\":\"%d\"}",
       _value, timestr, _error_count) ;
reset_watch_dog() ;
    result = afero->setAttribute(1, _str_buf) ;
    return( result == afSUCCESS ) ;
}

int v2x(float value)
{
    int result ;
    if (value < 0) {
        result = 20 ;
    } else if (value > 4) {
        result = 300 ;
    } else {
        result = 20 + (int)(70 * value + 0.5) ;
    }
    return( result ) ;
}

int press_v2y(float value, edge_chart_type *p)
{
    int y ;
    if (value < p->min) {
        value = p->min ;
    } else if (value > p->max) {
        value = p->max ;
    }
    y = p->top + p->height - 2
        - (int)((p->height - 2) * ((value - p->min) /(p->max - p->min))) ;
    return( y ) ;
}

/**
 * drawPointer
 *
 * draw a triangle pointer at value place
 * in GAS pressure display mode 
 */
void edge_pressure::drawPointer(int c)
{
    float delta_x ;
    int x[2], y, i ;
    const int top = 75 ;
    const int pointer_height = 15 ;
    for (i = 0 ; i < pointer_height ; i++ ) {
        y = top + i ;
        delta_x = i * 5.0 / 8.0 ;
        x[0] = c - delta_x ;
        x[1] = c + delta_x ;
        display->line(x[0], y, x[1], y, White) ;
    }
}

void edge_pressure::show(void)
{
    edge_chart_type *p = &edge_chart[ _id ] ;
    uint16_t color = White ;
    char str_buf[32] ;
    int i, x, y, l, r, c, str_x ;
    if (display) {
        reset_watch_dog() ;
/* for debug */
//  _value = _lower - 0.5 ; /* LOW */
// _value = (_higher + _lower) / 2 ; /* GOOD */
// _value = _higher + 0.2 ; /* HIGH */
 
        if (_value > _higher) {
            sprintf(str_buf, "HIGH") ;
            color = Red ;
            str_x = 60 ;
        } else if (_value < _lower) {
            sprintf(str_buf, "LOW") ;
            color = Yellow ;
            str_x = 60 ;
        } else {
            sprintf(str_buf, "GOOD") ;
            color = Green ;
            str_x = 35 ;
        }
        switch(display_mode) {
        case DISPLAY_MODE_GAS:
            display->BusEnable(true) ;
            display->cls() ;
            /* printf frame */
            display->foreground(White) ;
            display->line(20, 75, 300, 75, White) ;
            for (i = 0 ; i <= 8 ; i++ ) {
                x = 20 + i * 35 ;
                if (i & 0x01) { /* odd */
                    display->line(x, 55, x, 95, White) ;
                } else { /* even */
                    display->line(x, 45, x, 105, White) ;
                }
            }
            display->set_font((unsigned char*) Arial28x28);
            for (i = 0 ; i <= 4 ; i++ ) {
                x = 12 + i * 70 ;
                display->locate(x, 10) ;
                display->printf("%d", i) ;
            }
            /* print expected area and current pressure */
            l = v2x(_lower) ;
            r = v2x(_higher) ;
            c = v2x(_value) ;
//            display->fillrect(l, 70, r, 80, Red) ;
            display->fillrect(l, 65, r, 74, Red) ;
//            display->fillcircle(c, 75, 10, White) ;
            drawPointer(c) ;
                    
            /* print status */
            display->locate(str_x, 140) ;
            display->set_font_zoom(3, 3) ;
            display->foreground(color) ;
            display->printf(str_buf) ;
            display->set_font_zoom(1, 1) ;
            display->BusEnable(false) ;
            break ;
        case DISPLAY_MODE_SUMMARY:
            display->BusEnable(true) ;
            display->set_font((unsigned char*) Arial12x12);
            display->set_font_zoom(2, 2) ;
            display->foreground(White) ;
            display->locate(10, EDGE_SUMMARY_TIME_Y) ;
            displayTime(_sampled_time) ;
//          display->locate(10,50) ;
//          display->printf(timestr) ;
            display->locate(10, EDGE_SUMMARY_PRESS_Y) ;
            display->printf("Press: ") ;
            display->foreground(color) ;
            display->locate(90, EDGE_SUMMARY_PRESS_Y) ;
            display->printf("%.3f ", _value) ;
            display->foreground(White) ;
            display->printf("kgf/cm2") ;
            display->BusEnable(false) ;
            break ;
        case DISPLAY_MODE_CHART:
            x = p->left + p->index + 1;
            y = press_v2y(_value, p) ;
            display->BusEnable(true) ;
            if (p->index == 0) {
                draw_chart_frame(p) ;
            }
            display->foreground(color) ;
            display->pixel(x, y, color) ;
            display->set_font((unsigned char*) Arial12x12);
            display->locate(p->left + 40, p->top + 5) ;
            display->printf("%5s", str_buf) ;
            display->foreground(White) ;
            display->BusEnable(false) ;
            p->index = (p->index + 1) % (p->width - 2) ;
            break ;
        }
    }
    reset_watch_dog() ;
}
