#include "mbed.h"
#include "afLib.h"
#include "edge_reset_mgr.h"
#include "edge_sensor.h"
#include "edge_accel.h"
#include "MMA8451Q.h"

edge_accel::edge_accel(MMA8451Q *accel) : edge_sensor() 
{
    _accel = accel ;
    _sample_count = 0 ;
    _accumulation = 0 ;
    _prev_x = 0 ;
    _prev_y = 0 ;
    _prev_z = 0 ;
    
    _interval = 30 ;
}

edge_accel::~edge_accel(void)
{
    delete _accel ;
}

void    edge_accel::reset(void) 
{
    clear_value() ;
    edge_sensor::reset() ;
}

#if 0
void    edge_accel::prepare(void) 
{
//    printf("accel prepare\n") ;
}
#endif

int    edge_accel::sample(void) 
{
    int result ;
    float theValue = 0.0 ;
    if (_sample_count > 1) { /* if sample is 1 or less, no data */
        _num_sampled = _sample_count - 1 ;
        theValue = (float)_accumulation / (float)(_num_sampled) ;
        result = 0 ; /* success */
    } else {
        result = 1 ; /* fail! */
    }
    _value = theValue / 4096.0 ;
    _sampled_time = edge_time ;
    _sample_count = 0 ;
    _accumulation = 0 ;
    return( result ) ;
}

int    edge_accel::deliver(void) 
{
    int result ;
    char timestr[16] ;
    
    print_time(_sampled_time) ;
//    _value = get_value() ;
    printf(" accel: %.3f [%d samples]\n", _value, _num_sampled) ;
    time2seq(_sampled_time, timestr) ;
    sprintf(_str_buf,
        "{\"DEVICE\":\"ACCEL\",\"PN\":\"MMA8451Q\",\"VAL_X\":\"%.3f\",\"VAL_Y\":\"0\",\"VAL_Z\":\"0\",\"UNIT\":\"g\",\"T\":\"%s\",\"E\":\"%d\"}",
        _value, timestr, _error_count) ;
    result = afero->setAttribute(1, _str_buf) ;

    return( result == afSUCCESS ) ;
}

int accel_v2y(float value, edge_chart_type *p)
{
    int y ;
    if (value < p->min) {
        value = p->min ;
    } else if (value > p->max) {
        value = p->max ;
    }
    y = p->top + p->height/2 - 1
        - (int)((p->height - 2) * value /(p->max - p->min)) ;
    return( y ) ;
}

void edge_accel::show(void)
{
    int x, y ;
    edge_chart_type *p = &edge_chart[0] ; /* edge_chart for accel */
    reset_watch_dog() ;
    if (display) {
        switch(display_mode) {
        case DISPLAY_MODE_SUMMARY:
            display->BusEnable(true) ;
            display->set_font((unsigned char*) Arial12x12);
            display->set_font_zoom(2, 2) ;
            display->foreground(White) ;
            display->locate(EDGE_SUMMARY_X, EDGE_SUMMARY_TIME_Y) ;
            displayTime(_sampled_time) ;
//          display->locate(10, 5) ;
//          display->printf(timestr) ;
            display->locate(EDGE_SUMMARY_X, EDGE_SUMMARY_ACCEL_Y) ;
            display->printf("Accel: %.3f [%4d]", _value, _num_sampled) ;
            display->BusEnable(false) ;
            reset_watch_dog() ;
            break ;
        case DISPLAY_MODE_CHART:
            x = p->left + p->index + 1;
            y = accel_v2y(_value, p) ;
            display->BusEnable(true) ;
            if (p->index == 0) {
                draw_chart_frame(p) ;
            }
            display->pixel(x, y, White) ;
            display->BusEnable(false) ;
            p->index = (p->index + 1) % (p->width - 2) ;
            break ;
        default:
            break ;
        }
    }
    clear_value() ;
    reset_watch_dog() ;
}

int edge_accel::accum(void)
{
    int result ;
    int16_t value[3] ;

    if (_enable) {
       result = _accel->getAllRawData(value) ;
       
        if (result == 0) { /* success */
            if (_sample_count != 0) { /* first data does not have prev_data */
                    _accumulation +=
                    abs(_prev_x - value[0])
                    + abs(_prev_y - value[1])
                    + abs(_prev_z - value[2]) ; 
            }
            
            _sample_count++ ;
    
            _prev_x = value[0] ;
            _prev_y = value[1] ;
            _prev_z = value[2] ;
        }
    }
        
    return( result ) ;
}

void edge_accel::clear_value(void)
{
    _sample_count = 0 ;
    _accumulation = 0 ;
    _prev_x = 0 ;
    _prev_y = 0 ;
    _prev_z = 0 ;
}


        