#ifndef _LM75B_H_
#define _LM75B_H_

#include "mbed.h"

/**
* NXP LM75B Digital temperature sensor and thermal watchdog
*
* @code
#include "mbed.h"
#include "LM75B.h"
#define LM75B_I2C_ADDRESS (0x48)

#if defined (TARGET_KL25Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL46Z)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_K64F)
#define PIN_SCL  PTE24
#define PIN_SDA  PTE25
#elif defined (TARGET_K22F)
#define PIN_SCL  PTE1
#define PIN_SDA  PTE0
#elif defined (TARGET_KL05Z)
#define PIN_SCL  PTB3
#define PIN_SDA  PTB4
#elif defined (TARGET_NUCLEO_F411RE)
#define PIN_SCL  PB_8
#define PIN_SDA  PB_9
#else
 #error TARGET NOT DEFINED
#endif

int main() {
    int8_t itemp = 0 ;
    float ftemp = 0.0 ;
    LM75B lm75b(PIN_SDA, PIN_SCL, LM75B_I2C_ADDRESS) ;
    
    while(1) {
        itemp = lm75b.temp() ;
        lm75b.getTemp(&ftemp) ;
        printf("Temp = %d C degree,  %.3f C degree\n", itemp, ftemp) ;
        wait(1) ;
    }
}
* @endcode
*/
class LM75B
{
public:
  /**
  * LM75B constructor
  *
  * @param i2c pointer to the I2C object
  * @param addr addr of the I2C peripheral
  */
  LM75B(I2C *i2c, int addr);

  /**
  * LM75B destructor
  */
  ~LM75B();

  /**
  * get temperature as one byte (signed)
  * @param *temp int8_t returns integer part of the temperature
  * @return 0: success not-0: failure
  */
  int temp(int8_t *temp) ;
  
  /**
   * get temperature as 11 bit (float)
   * @param *temp float returns the temperature as float 
   * @return 0: success not-0: failure
   */
  int getTemp(float *temp) ;
  
  /**
   * get configuration register
   * @param ptr_byte uint8_t pointer value for the register
   * @param *config_data uint8_t value of the config register
   * @return 0: success non-0: failure
   */
  int getConfig(uint8_t ptr_byte, uint8_t *config_data) ;
  
  /**
   * set configuration register
   * @param ptr_byte uint8_t pointer value for the register
   * @param config_data uint8_t value to set in the config register
   * @return 0: success non-0: failure
   */
  int setConfig(uint8_t ptr_byte, uint8_t config_data) ;

private:
  I2C *p_i2c;
  int m_addr;
  int readRegs(int addr, uint8_t * data, int len);
  int writeRegs(uint8_t * data, int len);

};

#endif