
/**
 ******************************************************************************
 * @file    main.cpp
 * @date    February 06, 2020
 * @brief   mbed test application - Esmacat Shield(EASE) working together with 
 *          Base Board with Arduino UNO form factor as EtherCAT slave.
 *          With the successful execution of this code the LED on EASE should
 *          blink. It should also estabilish data transfer between the EtherCAT
 *          Master of Esmacat on the PC.
 *          For further information please refer to the tutorials at
 *          https://www.esmacat.com/tutorials 
 ******************************************************************************
 
  Copyright (c) 2020 https://www.esmacat.com/

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  EsmacatShield.h - Library for using EtherCAT Arduino Shield by Esmacat(EASE).
  Created by Esmacat, 01/22/2020
  
*******************************************************************************
* @file EsmacatShield.h
*******************************************************************************
*/
#include "mbed.h"
//#include "BufferedSerial.h"
#include <EsmacatShield.h> //Include EsmacatShield Library
#include "keypad.h"


int counter;
int16_t v[8];              //an array of integer is declared for reading the 
                           //data packet of EtherCAT from EASE 8 registers
                           
//BufferedSerial pc(PA_2, PA_3, 9600);   // Configuring the serial port to host PC
Ticker tick1;

SPI spi(D11, D12, D13);             // Infineon <->ARM mosi, miso, sclk
DigitalOut selectPin(D10);          // D10 is used to drive chip enable low


SPI spi2(PB_15, PB_14, PB_13);      // iC-MB4 Encoder1 & Encoder2 <->ARM mosi, miso, sclk
SPI spi3(PC_12, PC_11, PC_10);      // iC-MB4 Encoder1 & Encoder2 <->ARM mosi, miso, sclk
DigitalOut CS1(PA_8);               //iC-MB4 Encoder1 CS1;
DigitalOut CS2(PB_1);               //iC-MB4 Encoder2 CS2;
DigitalOut CS3(PA_13);              //iC-MB4 Encoder2 CS3;
DigitalOut CS4(PA_14);               //iC-MB4 Encoder2 CS4;

DigitalOut CS_ENC(PC_0);

DigitalOut NRES1(PH_1); 
DigitalOut NRES2(PH_0);          
DigitalOut NRES3(PC_15);
DigitalOut NRES4(PC_14);  
                    
uint32_t Encoder2_Raw;
uint32_t Encoder2_Filter;
uint32_t Encoder2_Data;

uint32_t Encoder3_Raw;
uint32_t Encoder3_Filter;
uint32_t Encoder3_Data;

uint32_t Encoder1_Raw;
uint32_t Encoder1_Filter;
uint32_t Encoder1_Data;

uint32_t Encoder4_Raw;
uint32_t Encoder4_Filter;
uint32_t Encoder4_Data;


uint16_t E1_HIGH;
uint16_t E1_LOW;
uint16_t E2_HIGH;
uint16_t E2_LOW;
uint16_t E3_HIGH;
uint16_t E3_LOW;
uint16_t E4_HIGH;
uint16_t E4_LOW;

uint8_t  Hbyte_enc = 0;
uint8_t  Lbyte_enc = 0;
uint16_t Angle_encoder = 0;

#define WRITE_DATA   0x02
#define READ_DATA    0x03
#define READ_STATUS  0x05
#define WRITE_INSTRUCTION 0x07

#define CHSEL     0xE4
#define REGVERS   0xE5 
#define FREQ      0xE6
#define CFGCH1    0xED
#define FREQAGS   0xE8
#define BREAK     0x80
#define AGSFREQ   0x81
#define CFGIF     0xF5
#define SCDLEN1   0xC0
#define SELCRCS1  0xC1
#define SCRCSTART1_L 0xC2
#define SCRCSTART1_H 0xC3
#define ACTnSENS  0xEF
#define INSTR     0xF4

uint8_t Cmd;
uint8_t CFG_Data[16];
uint8_t Read_Data2[32];
uint8_t Read_Data3[32];
uint8_t Read_Data1[32];
uint8_t Read_Data4[32];

uint8_t RegisterData_RDATA = 0x80;



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
void SetSPI(SPI&spi)
{
  spi.format(8, 0);
  spi.frequency(1000000);  
}


void iC_MB4_WriteInstruction_BREAK(SPI&spi, DigitalOut &CS)
{
  /*
  Tell Master to stop any previous and start fresh
  */
  CS = 0;
  wait_us(1);
  
  spi.write(WRITE_INSTRUCTION);
  spi.write(BREAK);              
  
  CS = 1;    
}


void iC_MB4_WriteRegister_CHSEL(SPI&spi, DigitalOut &CS)
{
  /*
  Set the Channel 1 as the only active channel
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(CHSEL);   //0xE4
  spi.write(0x01);    
  
  CS = 1;    
}


void iC_MB4_WriteRegister_REGVERS(SPI&spi, DigitalOut &CS)
{
  /*
  Set up the channel as BiSS register configuration
  The iC-MB4 does support autonomous register communication with BiSS B and BiSS C.
  0 - BiSS A/B
  1 - BiSS C
  */
  uint8_t temp;
  temp = 0x01 << 6;  //Command/instruction communication
  //temp = 0xC0;     //Register access
  
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(REGVERS);  //0xE5
  spi.write(temp);     //h0x40 b0100 0000 
  
  CS = 1;    
}


void iC_MB4_WriteRegister_FREQ(SPI&spi, DigitalOut &CS)
{
  /*
  Set the FREQ register bit 4:0 to communicate with encoder
  Set 0.5 MHz
  RLS - BiSS Mode Clock frequency range: 150kHz to 5 MHz
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(FREQ);     //0xE6
  spi.write(0x11);     //
  
  CS = 1;    
}


void iC_MB4_WriteRegister_BiSS_C(SPI&spi, DigitalOut &CS)
{
  /*
  Set up the communicaton for BiSS_C protocol
  0x00 - BiSS B
  0x01 - BiSS C
  0x02 - SSI
  0x03 - Channel is not used
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(CFGCH1);
  spi.write(0x01);
  
  CS = 1;    
}


void iC_MB4_WriteRegister_BiSS_frame_rate(SPI&spi, DigitalOut &CS)
{
  /*
  Set up for automatically starting read cycles
  The FREQAGS controls the automatic data transmission (AutoGetSens) enabled by the instruction bit AGS
  Set sampling rate 16kHz
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(FREQAGS);    //0xE8
  spi.write(0x7C);       //AGSMIN
  //spi.write(AGSFREQ);    //
  
  CS = 1;    
}


void iC_MB4_WriteRegister_RS422Mode(SPI&spi, DigitalOut &CS)
{
   /*
  Set up for RS422 line level
  */
  uint8_t temp;
  temp = 0x09;
  
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(CFGIF);
  spi.write(temp);
  
  CS = 1;    
}


void iC_MB4_WriteRegister_SetSCD(SPI&spi, DigitalOut &CS)
{
  /*
  Configure the data length of the SCD.
  Resolution: MB029 17B
  -> 17bits of position + 2 status bits(Error+Warning) + 6 bits CRC = 25 bits long data packet.
  Resolution: MB039 19B
  -> 19bits of position + 2 status bits(Error+Warning) + 6 bits CRC = 27 bits long data packet.
  */
  uint8_t temp;
  uint8_t temp1;
  uint8_t temp2;
  temp = 0x22 |0x40;  //24b
  temp1 = 0x19 |0x40;  //25bits
  temp2 = 0x1B |0x40;  //27bits
  
  
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(SCDLEN1);   //0xC0
  spi.write(temp1);      
  
  CS = 1;    
}


void iC_MB4_WriteRegister_SetSCD2(SPI&spi, DigitalOut &CS)
{
  /*
  Configure the data length of the SCD.
  Resolution: MB029 17B
  -> 17bits of position + 2 status bits(Error+Warning) + 6 bits CRC = 25 bits long data packet.
  Resolution: MB039 19B
  -> 19bits of position + 2 status bits(Error+Warning) + 6 bits CRC = 27 bits long data packet.
  */
  uint8_t temp;
  uint8_t temp1;
  uint8_t temp2;
  temp = 0x22 |0x40;  //24b
  temp1 = 0x19 |0x40;  //25bits
  temp2 = 0x1B |0x40;  //27bits
  
  
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(SCDLEN1);   //0xC0
  spi.write(temp);      
  
  CS = 1;    
}


void iC_MB4_WriteRegister_SELCRCS(SPI&spi, DigitalOut &CS)
{
  /*
  Configure the CRC info
  */
  uint8_t temp;
  temp = (0x00 << 7) |0x06; 
  //temp = (0x01 << 7) |0x21; 
  
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(SELCRCS1);  //0xC1
  spi.write(temp);      //Code: 6 
  
  CS = 1;    
}


void iC_MB4_WriteRegister_SCRCSTART1_L(SPI&spi, DigitalOut &CS)
{
  /*
  Configure the CRC start
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(SCRCSTART1_L);
  spi.write(0x00);
  
  CS = 1;    
}


void iC_MB4_WriteRegister_SCRCSTART1_H(SPI&spi, DigitalOut &CS)
{
  /*
  Configure the CRC start
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(SCRCSTART1_H);
  spi.write(0x00);
  
  CS = 1;    
}


void iC_MB4_WriteRegister_ACTnSENS(SPI&spi, DigitalOut &CS)
{
  /*
  Configure all slaves to be sensors
  */
  CS = 0;
  wait_us(10);
  
  spi.write(WRITE_DATA);
  spi.write(ACTnSENS);
  spi.write(0x00);
  
  CS = 1;    
}


void iC_MB4_WriteInstruction_INSTR(SPI&spi, DigitalOut &CS)
{
  /*
  Enable the AGS (Automatic Get Sensor) bit so that the iC-MB4 now polls
  
  */
  CS = 0;
  wait_us(10);
  uint8_t temp;
  
//  spi.write(READ_DATA);
//  spi.write(0xF4);
//  temp = spi.write(0x00);
  
//  wait_us(10);
  temp = temp | 0x01;
  
  spi.write(0x07);
  //spi.write(0xF4);  //Instruction Register 0xF4
  spi.write(0x01);
  
  CS = 1;    
}


void ReadiCMB4(SPI&spi,uint8_t cmd, uint8_t &data, DigitalOut& CS)
{
  CS = 0;
  wait_us(10);
  
  spi.write(READ_DATA);
  spi.write(cmd);
  data = spi.write(0x00);
  
  CS = 1;
}


void iCMB4_Read_SensorData(SPI&spi, uint8_t Read_Data[], DigitalOut& CS)
{
  CS = 0;
  wait_us(10);
  
  spi.write(READ_DATA); 
  spi.write(0x00);   //SCDATA Addr 0x00 전송
  Read_Data[0] = spi.write(0x01); //SCDATA Addr 0x01 전송 & 0x00번지의 값 읽음
  Read_Data[1] = spi.write(0x02); //SCDATA Addr 0x02 전송 & 0x01번지의 값 읽음
  Read_Data[2] = spi.write(0x03); //0x02번지의 값 읽음
  Read_Data[3] = spi.write(0x04);
  Read_Data[4] = spi.write(0x05);
  Read_Data[5] = spi.write(0x06);
  Read_Data[6] = spi.write(0x07);
  Read_Data[7] = spi.write(0x00);
  
  CS = 1;
  CS = 1;
}

//only camera robot
void ReadEncoder(SPI&spi, uint8_t &low, uint8_t &high, DigitalOut& CS, uint16_t &angle)
{
  CS = 0;
  high = spi.write(0x00);
  low = spi.write(0x00);
  CS = 1;

  uint16_t Data = ((uint16_t)high << 2) + ((uint16_t)low >>6); 
  high = Data >> 8;
  low = Data & 0xff;
  //angle = (float)Data * (360.0 / 1024.0);
  angle = (uint16_t) angle * 1;
}

//////////////////////////////////////////////////////////////////////////////// 
////////////////////////////////////////////////////////////////////////////////
int main() 
{
    CS1 = 1;
    CS2 = 1;
    CS3 = 1;
    CS4 = 1;                              //SPI CS HIGH
    //CS_ENC = 1;
    
    EsmacatShield slave(spi, selectPin);
    slave.setup_spi();
    
    SetSPI(spi2);                         //Setup SPI for iC-MB4
    SetSPI(spi3);
     
  
    
    wait_us(100000);   
    
    
    //iC-MB4 Configuration
    iC_MB4_WriteInstruction_BREAK(spi3, CS3);
    Cmd = 0xF4;
    ReadiCMB4(spi3, Cmd, CFG_Data[0], CS3);
     

    iC_MB4_WriteRegister_CHSEL(spi3, CS3);    
    Cmd = CHSEL;
    ReadiCMB4(spi3, Cmd, CFG_Data[1], CS3);

    
    iC_MB4_WriteRegister_REGVERS(spi3, CS3);   
    Cmd = REGVERS;
    ReadiCMB4(spi3, Cmd, CFG_Data[2], CS3);
  
    
    iC_MB4_WriteRegister_FREQ(spi3, CS3);   
    Cmd = FREQ;
    ReadiCMB4(spi3, Cmd, CFG_Data[3], CS3);
     
    
    iC_MB4_WriteRegister_BiSS_C(spi3, CS3); 
    Cmd = CFGCH1;
    ReadiCMB4(spi3, Cmd, CFG_Data[4], CS3);
      
    
    iC_MB4_WriteRegister_BiSS_frame_rate(spi3, CS3); 
    Cmd = FREQAGS;
    ReadiCMB4(spi3, Cmd, CFG_Data[5], CS3);
       
    
    iC_MB4_WriteRegister_RS422Mode(spi3, CS3);  
    Cmd = CFGIF;
    ReadiCMB4(spi3, Cmd, CFG_Data[6], CS3);
       
    
    iC_MB4_WriteRegister_SetSCD2(spi3, CS3);  
    Cmd = SCDLEN1;
    ReadiCMB4(spi3, Cmd, CFG_Data[7], CS3);
        
    
    iC_MB4_WriteRegister_SELCRCS(spi3, CS3);
    Cmd = SELCRCS1;
    ReadiCMB4(spi3, Cmd, CFG_Data[8], CS3);
       
    
    iC_MB4_WriteRegister_SCRCSTART1_L(spi3, CS3);    
    Cmd = SCRCSTART1_L;
    ReadiCMB4(spi3, Cmd, CFG_Data[9], CS3);
      
    
    iC_MB4_WriteRegister_SCRCSTART1_H(spi3, CS3); 
    Cmd = SCRCSTART1_H;
    ReadiCMB4(spi3, Cmd, CFG_Data[10], CS3);
    
    
    iC_MB4_WriteRegister_ACTnSENS(spi3, CS3);   
    Cmd = ACTnSENS;
    ReadiCMB4(spi3, Cmd, CFG_Data[11], CS3);
     
    
    iC_MB4_WriteInstruction_INSTR(spi3, CS3);
   
    Cmd = INSTR;
    ReadiCMB4(spi3, Cmd, CFG_Data[12], CS3);    

    //pc.write(CFG_Data, 13);
    //wait_us(10000);    

   
    //iC-MB4 Configuration
    iC_MB4_WriteInstruction_BREAK(spi2, CS2);
    Cmd = 0xF4;
    ReadiCMB4(spi2, Cmd, CFG_Data[0], CS2);
     

    iC_MB4_WriteRegister_CHSEL(spi2, CS2);    
    Cmd = CHSEL;
    ReadiCMB4(spi2, Cmd, CFG_Data[1], CS2);

    
    iC_MB4_WriteRegister_REGVERS(spi2, CS2);   
    Cmd = REGVERS;
    ReadiCMB4(spi2, Cmd, CFG_Data[2], CS2);
  
    
    iC_MB4_WriteRegister_FREQ(spi2, CS2);   
    Cmd = FREQ;
    ReadiCMB4(spi2, Cmd, CFG_Data[3], CS2);
     
    
    iC_MB4_WriteRegister_BiSS_C(spi2, CS2); 
    Cmd = CFGCH1;
    ReadiCMB4(spi2, Cmd, CFG_Data[4], CS2);
      
    
    iC_MB4_WriteRegister_BiSS_frame_rate(spi2, CS2); 
    Cmd = FREQAGS;
    ReadiCMB4(spi2, Cmd, CFG_Data[5], CS2);
       
    
    iC_MB4_WriteRegister_RS422Mode(spi2, CS2);  
    Cmd = CFGIF;
    ReadiCMB4(spi2, Cmd, CFG_Data[6], CS2);
       
    
    iC_MB4_WriteRegister_SetSCD(spi2, CS2);  
    Cmd = SCDLEN1;
    ReadiCMB4(spi2, Cmd, CFG_Data[7], CS2);
        
    
    iC_MB4_WriteRegister_SELCRCS(spi2, CS2);
    Cmd = SELCRCS1;
    ReadiCMB4(spi2, Cmd, CFG_Data[8], CS2);
       
    
    iC_MB4_WriteRegister_SCRCSTART1_L(spi2, CS2);    
    Cmd = SCRCSTART1_L;
    ReadiCMB4(spi2, Cmd, CFG_Data[9], CS2);
      
    
    iC_MB4_WriteRegister_SCRCSTART1_H(spi2, CS2); 
    Cmd = SCRCSTART1_H;
    ReadiCMB4(spi2, Cmd, CFG_Data[10], CS2);
    
    
    iC_MB4_WriteRegister_ACTnSENS(spi2, CS2);   
    Cmd = ACTnSENS;
    ReadiCMB4(spi2, Cmd, CFG_Data[11], CS2);
     
    
    iC_MB4_WriteInstruction_INSTR(spi2, CS2);
   
    Cmd = INSTR;
    ReadiCMB4(spi2, Cmd, CFG_Data[12], CS2);    

    wait_us(1000);

    //iC-MB4 Configuration
    iC_MB4_WriteInstruction_BREAK(spi2, CS1);
    Cmd = 0xF4;
    ReadiCMB4(spi2, Cmd, CFG_Data[0], CS1);
     

    iC_MB4_WriteRegister_CHSEL(spi2, CS1);    
    Cmd = CHSEL;
    ReadiCMB4(spi2, Cmd, CFG_Data[1], CS1);

    
    iC_MB4_WriteRegister_REGVERS(spi2, CS1);   
    Cmd = REGVERS;
    ReadiCMB4(spi2, Cmd, CFG_Data[2], CS1);
  
    
    iC_MB4_WriteRegister_FREQ(spi2, CS1);   
    Cmd = FREQ;
    ReadiCMB4(spi2, Cmd, CFG_Data[3], CS1);
     
    
    iC_MB4_WriteRegister_BiSS_C(spi2, CS1); 
    Cmd = CFGCH1;
    ReadiCMB4(spi2, Cmd, CFG_Data[4], CS1);
      
    
    iC_MB4_WriteRegister_BiSS_frame_rate(spi2, CS1); 
    Cmd = FREQAGS;
    ReadiCMB4(spi2, Cmd, CFG_Data[5], CS1);
       
    
    iC_MB4_WriteRegister_RS422Mode(spi2, CS1);  
    Cmd = CFGIF;
    ReadiCMB4(spi2, Cmd, CFG_Data[6], CS1);
       
    
    iC_MB4_WriteRegister_SetSCD(spi2, CS1);  
    Cmd = SCDLEN1;
    ReadiCMB4(spi2, Cmd, CFG_Data[7], CS1);
        
    
    iC_MB4_WriteRegister_SELCRCS(spi2, CS1);
    Cmd = SELCRCS1;
    ReadiCMB4(spi2, Cmd, CFG_Data[8], CS1);
       
    
    iC_MB4_WriteRegister_SCRCSTART1_L(spi2, CS1);    
    Cmd = SCRCSTART1_L;
    ReadiCMB4(spi2, Cmd, CFG_Data[9], CS1);
      
    
    iC_MB4_WriteRegister_SCRCSTART1_H(spi2, CS1); 
    Cmd = SCRCSTART1_H;
    ReadiCMB4(spi2, Cmd, CFG_Data[10], CS1);
    
    
    iC_MB4_WriteRegister_ACTnSENS(spi2, CS1);   
    Cmd = ACTnSENS;
    ReadiCMB4(spi2, Cmd, CFG_Data[11], CS1);
     
    
    iC_MB4_WriteInstruction_INSTR(spi2, CS1);
   
    Cmd = INSTR;
    ReadiCMB4(spi2, Cmd, CFG_Data[12], CS1);    

    //pc.write(CFG_Data, 13);
    //wait_us(10000); 
   
   
    //iC-MB4 Configuration
    iC_MB4_WriteInstruction_BREAK(spi3, CS4);
    Cmd = 0xF4;
    ReadiCMB4(spi3, Cmd, CFG_Data[0], CS4);
     

    iC_MB4_WriteRegister_CHSEL(spi3, CS4);    
    Cmd = CHSEL;
    ReadiCMB4(spi3, Cmd, CFG_Data[1], CS4);

    
    iC_MB4_WriteRegister_REGVERS(spi3, CS4);   
    Cmd = REGVERS;
    ReadiCMB4(spi3, Cmd, CFG_Data[2], CS4);
  
    
    iC_MB4_WriteRegister_FREQ(spi3, CS4);   
    Cmd = FREQ;
    ReadiCMB4(spi3, Cmd, CFG_Data[3], CS4);
     
    
    iC_MB4_WriteRegister_BiSS_C(spi3, CS4); 
    Cmd = CFGCH1;
    ReadiCMB4(spi3, Cmd, CFG_Data[4], CS4);
      
    
    iC_MB4_WriteRegister_BiSS_frame_rate(spi3, CS4); 
    Cmd = FREQAGS;
    ReadiCMB4(spi3, Cmd, CFG_Data[5], CS4);
       
    
    iC_MB4_WriteRegister_RS422Mode(spi3, CS4);  
    Cmd = CFGIF;
    ReadiCMB4(spi3, Cmd, CFG_Data[6], CS4);
       
    
    iC_MB4_WriteRegister_SetSCD2(spi3, CS4);  
    Cmd = SCDLEN1;
    ReadiCMB4(spi3, Cmd, CFG_Data[7], CS4);
        
    
    iC_MB4_WriteRegister_SELCRCS(spi3, CS4);
    Cmd = SELCRCS1;
    ReadiCMB4(spi3, Cmd, CFG_Data[8], CS4);
       
    
    iC_MB4_WriteRegister_SCRCSTART1_L(spi3, CS4);    
    Cmd = SCRCSTART1_L;
    ReadiCMB4(spi3, Cmd, CFG_Data[9], CS4);
      
    
    iC_MB4_WriteRegister_SCRCSTART1_H(spi3, CS4); 
    Cmd = SCRCSTART1_H;
    ReadiCMB4(spi3, Cmd, CFG_Data[10], CS4);
    
    
    iC_MB4_WriteRegister_ACTnSENS(spi3, CS4);   
    Cmd = ACTnSENS;
    ReadiCMB4(spi3, Cmd, CFG_Data[11], CS4);
     
    
    iC_MB4_WriteInstruction_INSTR(spi3, CS4);
   
    Cmd = INSTR;
    ReadiCMB4(spi3, Cmd, CFG_Data[12], CS4);


    //pc.write(CFG_Data, 13);
    wait_us(10000); 
   



    while(1)
    {
      iCMB4_Read_SensorData(spi2,Read_Data1, CS1);
      iCMB4_Read_SensorData(spi2,Read_Data2, CS2);
      iCMB4_Read_SensorData(spi3,Read_Data3, CS3);  //camera robot -> absolute encoder
      iCMB4_Read_SensorData(spi3,Read_Data4, CS4);
     
      //only camera robot
      //ReadEncoder(spi3, Lbyte_enc, Hbyte_enc, CS_ENC, Angle_encoder);
      
      //Encoder circle1
      Encoder2_Raw = ((Read_Data2[3] << 24)| (Read_Data2[2] << 16) | (Read_Data2[1] << 8) | (Read_Data2[0]));
      Encoder2_Filter = Encoder2_Raw >> 9;
      //Encoder2_Data = (((unsigned long)Encoder2_Filter * 360 * 10000) / 0x1FFFF);  // range: 0~360 degree , 0.001 degree
      float a = (((float)Encoder2_Filter * 360 * 10000) / 0x1FFFF);  // range: 0~360 degree , 0.001 degree
      Encoder2_Data=a;
      E2_HIGH = Encoder2_Data >> 16;
      E2_LOW = Encoder2_Data & 0xffff;
      if(Read_Data2[4]==0){
         //정상 상태
      }
      else {
        //엔코더 비정상 상태이므로, 최상위 비트를 1로 바꿔서 전달한다.
        E2_HIGH = 0x8000;   
      }
      
         
      //Encoder linear2
      Encoder3_Raw = ((Read_Data3[3] << 24)| (Read_Data3[2] << 16) | (Read_Data3[1] << 8) | (Read_Data3[0]));
      Encoder3_Filter = Encoder3_Raw >> 1;
      Encoder3_Data = Encoder3_Filter / 1000; // unit: 0.001 mm 
      E3_HIGH =  Encoder3_Data >> 16;
      E3_LOW = Encoder3_Data & 0xffff;
      if(Read_Data3[4]==0){
         //정상 상태
      }
      else {
        //엔코더 비정상 상태이므로, 최상위 비트를 1로 바꿔서 전달한다.
        E3_HIGH = 0x8000;   
      }

      
      //Encoder circle2
      Encoder1_Raw = ((Read_Data1[3] << 24)| (Read_Data1[2] << 16) | (Read_Data1[1] << 8) | (Read_Data1[0]));
      Encoder1_Filter = Encoder1_Raw >> 9;
      //Encoder1_Data = ((Encoder1_Filter * 360) / 0x1FFFF)*10000;  // range: 0~360 degree, 0.001 resolution
      float b = (((float)Encoder1_Filter * 360 * 10000) / 0x1FFFF);  // range: 0~360 degree , 0.001 degree
      Encoder1_Data = b;
      E1_HIGH =  Encoder1_Data >> 16;
      E1_LOW = Encoder1_Data & 0xffff;
      if(Read_Data1[4]==0){
         //정상 상태
      }
      else {
        //엔코더 비정상 상태이므로, 최상위 비트를 1로 바꿔서 전달한다.
        E1_HIGH = 0x8000;   
      }
      
      //Encoder linear1
      Encoder4_Raw = ((Read_Data4[3] << 24)| (Read_Data4[2] << 16) | (Read_Data4[1] << 8) | (Read_Data4[0]));
      Encoder4_Filter = Encoder4_Raw >> 1;
      Encoder4_Data = Encoder4_Filter / 1000; // unit: 0.001 mm 
      E4_HIGH =  Encoder4_Data >> 16;
      E4_LOW = Encoder4_Data & 0x0ffff;
      if(Read_Data4[4]==0){
         //정상 상태
      }
      else {
        //엔코더 비정상 상태이므로, 최상위 비트를 1로 바꿔서 전달한다.
        E4_HIGH = 0x8000;   
      }
    
      
      printf("Encoder2: %d, Encoder3: %d\n", Encoder2_Data, Encoder3_Data);
      printf("Encoder1: %d, Encoder4: %d\n", Encoder1_Data, Encoder4_Data);
      printf("%02X, %02X\n", E1_HIGH, E1_LOW);
      //printf("Encoder_States: %d, Encoder4: %d \n", Read_Data4[3], Encoder4_Data);
     /*
      printf("Encoder3: %d, [4]:%02X", Encoder3_Data, E3_HIGH);
      if(Read_Data3[4]==0){
         printf("Good Data: %d\n", Encoder3_Data);
      }
      else{
         printf("Error/n");
      }
     */  
      slave.write_reg_value(0,E2_HIGH, true);  // 완전 서클
      slave.write_reg_value(1,E2_LOW, true);
      slave.write_reg_value(2,E4_HIGH, true);  // 리니어1
      slave.write_reg_value(3,E4_LOW, true);
      slave.write_reg_value(4,E1_HIGH, true);  // 반원 
      slave.write_reg_value(5,E1_LOW, true);
      slave.write_reg_value(6,E3_HIGH, true);  // 리니어2
      slave.write_reg_value(7,E3_LOW, true);
      
      //slave.write_reg_value(6,Angle_encoder, true);
      //slave.write_reg_value(7,0x00, true);
      
      wait_us(10000);
      
      //printf("%d\n", Angle_encoder);

    }

}
