/******************************************************************
* Authors: Riley Barnard, Kamyar Izad, Alec Selfridge, Dylan Smith   
* Date:    12/12/2015
* Rev:     1.2
* Project: RTOS Light Animations
* Notes:   This is the slave processor and is responsible for 
*          updating it's RGB LED and replying to requests from the
*          master based on predefined codes. One ISR is used to 
*          check the joystick so we always get the most current
*          user input. 
*
******************************************************************/

#include "mbed.h"

#define initResponse 0x55
#define R_CODE 1
#define G_CODE 2
#define B_CODE 3

/*
****************************************
              Functions
****************************************    
*/
void packRGB(float v); 
void RGBpicker(void);
void RGBPush(void);
void LEDInit(void);
void RgbInit(void);
/*
****************************************
             Global Data
****************************************    
*/
int Rtemp    = 0;
int Gtemp    = 0;
int Btemp    = 0;
float r      = 0;
float g      = 0;
float b      = 0;
uint8_t tick = 0;
uint8_t rx   = 0;
/*
****************************************
               Objects
****************************************    
*/
// interrupt for RGB updates
Ticker tock;
// init LEDs
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
// active-low 5-axis joystick
// center, left_hi, right_hi, left_lo, right_lo
BusIn joystick(p31, p32, p37, p39, p38);
// master processor on SPI bus
SPISlave master(p11, p12, p13, p14); // mosi, miso, sclk, ssel
// R of RGB LED
PwmOut R(p25);
// G of RGB LED
PwmOut G(p28);
// B of RGB LED
PwmOut B(p26);
//Serial debug(USBTX, USBRX);
/*
****************************************
            Main Execution
****************************************    
*/
int main() {
    LEDInit();
    // prime SPI with init response
    master.reply(initResponse);
    RgbInit();
    tock.attach(RGBpicker, 0.1);
     
    // wait indefinitely for a connection to the master
    while(1){
        if(master.receive())
            break;
    }
    
    while(true) {
        // check if the master wants data
        if (master.receive()) {
            rx = master.read();
            //debug.printf("Received: %d\n\r", rx);
            // determine what the master wants based on codes
            switch(rx) 
            {
                // requesting R
                case R_CODE: packRGB(r); 
                    break;
                // requesting G
                case G_CODE: packRGB(g);
                    break;
                // requesting B
                case B_CODE: packRGB(b);
                    break;
                // do nothing by default
                default: break;    
            }
        }
    }
}
/*
****************************************
          Function Definitions
****************************************    
*/
// initialize and test RGB LED
void RgbInit()
{
    // 1KHz
    R.period(.001);
    R = 1.0;
    G.period(.001);
    G = 1.0;
    B.period(.001);
    B = 1.0;
    wait(0.4);
    r = 1.0; g = 1.0; b = 1.0;
    RGBPush();
    wait_ms(150);
    r = 0.0; g = 1.0; b = 1.0;
    RGBPush();
    wait_ms(150);
    r = .6; g = 0.0; b = 0.0;
    RGBPush();
    wait_ms(150);
    r = 1.0; g = 1.0; b = 0.0;
    RGBPush();
    wait_ms(150);
    r = 1.0; g = 0.0; b = 0.0;
    RGBPush();
    wait_ms(150);
    r = 1.0; g = 0.0; b = 1.0;
    RGBPush();
    wait_ms(150);
    r = 1.0; g = 1.0; b = 0.0;
    RGBPush();
    wait_ms(150);
    r = 0.0; g = 0.0; b = 1.0;
    RGBPush();
    wait_ms(150);
    r = 0.0; g = 1.0; b = 1.0;
    RGBPush();
    wait_ms(150);
    r = 1.0; g = 1.0; b = 1.0;
    RGBPush();
    wait_ms(150);
}

// functional test of 4 LEDs 
void LEDInit(){
    led1 = 1;
    wait(0.20);
    led2 = 1;
    wait(0.20);
    led3 = 1;
    wait(0.20);
    led4 = 1;
    wait(0.20);
    led1 = 0;
    wait(0.20);
    led2 = 0;
    wait(0.20);
    led3 = 0;
    wait(0.20);
    led4 = 0;
    wait(0.20);
}

void RGBpicker(void) 
{
    // tick prevents this from being called more than once
    // in a reasonable amount of time
    tick++;
    if(tick > 1) {
        tick = 0;
        return;
    }
    // left : R
    if(!(joystick[3])) {
        if(R.read() == 0.0)
            r = 1.0;
        r = r - .02;
    }
    // right : B
    if(!(joystick[2])) {
        if(B.read() == 0.0)
            b = 1.0;
        b = b - .02;
    }
    // dn : G
    if(!(joystick[4])) {
        if(G.read() == 0.0)
            g = 1.0;
        g = g - .02;
    }
    // up : Reset
    if(!(joystick[1])) {
        r = 1.0; g = 1.0; b = 1.0;
    }
    // center : confirm changes
    if(!joystick[0]) {
        RGBPush();
    }
}

// assigns RGB to PWM objects
void RGBPush(void)
{
    R = r; 
    G = g; 
    B = b;
}

// converts a color encoded as a float into it's integer version
void packRGB (float v){
    int tmp = int(v*255.0);
    master.reply(tmp);
}